from itertools import chain
from collections import namedtuple


ServerVersion = namedtuple("ServerVersion", ("major", "minor", "basever"))


def generate_versions_description(deployed_server_versions, deployed_cs_versions):
    """
    >>> print(generate_versions_description({ServerVersion(123, 1, 1000): ['service_1', 'service_2']}, {'1': [ServerVersion(123, 1, 1000)]}))
    Frontend versions:
      123-1.1000 (2 services)
    CS versions:
      namespace 1: [123-1.1000]
    >>> print(generate_versions_description({ServerVersion(123, 1, 1000): ['service_1']}, {'1': [ServerVersion(123, 1, 1000)], '2': [ServerVersion(123, 2, 2000)]}))
    Frontend versions:
      123-1.1000 (1 services)
    CS versions:
      namespace 1: [123-1.1000]
      namespace 2: [123-2.2000]
    >>> print(generate_versions_description({ServerVersion(123, 1, 1000): ['service_1']}, {'1': [ServerVersion(123, 1, 1000), ServerVersion(123, 2, 2000)]}))
    Frontend versions:
      123-1.1000 (1 services)
    CS versions:
      namespace 1: [123-1.1000, 123-2.2000]
    >>> print(generate_versions_description({ServerVersion(123, 1, 1000): [1, 2], ServerVersion(123, 2, 2000): [3]}, {'1': [ServerVersion(123, 1, 1000)], '2': [ServerVersion(123, 2, 2000)]}))
    Frontend versions:
      123-2.2000 (1 services)
      123-1.1000 (2 services)
    CS versions:
      namespace 1: [123-1.1000]
      namespace 2: [123-2.2000]
    """
    return 'Frontend versions:\n{}\nCS versions:\n{}'.format(
        "\n".join(
            [
                "  {v.major}-{v.minor}.{v.basever} ({count} services)".format(
                    v=v, count=len(services)
                ) for v, services in sorted(deployed_server_versions.items(), key=lambda x: len(x[1]))
            ]
        ),
        "\n".join(
            [
                "  namespace {}: [{}]".format(
                    namespace, ", ".join(
                        ["{v.major}-{v.minor}.{v.basever}".format(v=v) for v in versions]
                    )
                ) for namespace, versions in deployed_cs_versions.items()
            ]
        ),
    )


def versions_synchronized(deployed_server_versions, deployed_cs_versions):
    """
    >>> base_versions_synchronized({}, {})
    False
    >>> base_versions_synchronized(
    ...     {ServerVersion(123, 1, 4000): [1]},
    ...     {}
    ... )
    False
    >>> base_versions_synchronized(
    ...     {
    ...         ServerVersion(123, 1, 4000): [1],
    ...         ServerVersion(123, 2, 4000): [2, 3],
    ...     },
    ...     {}
    ... )
    False
    >>> base_versions_synchronized(
    ...     {
    ...         ServerVersion(123, 1, 4000): [1],
    ...     },
    ...     {
    ...         '1': [ServerVersion(123, 1, 4000)]
    ...     }
    ... )
    True
    >>> base_versions_synchronized(
    ...     {
    ...         ServerVersion(123, 1, 4000): [2, 3],
    ...         ServerVersion(123, 2, 4000): [1],
    ...     },
    ...     {
    ...         '1': [ServerVersion(123, 1, 4000)]
    ...     }
    ... )
    True
    >>> base_versions_synchronized(
    ...     {
    ...         ServerVersion(123, 1, 4000): [1],
    ...     },
    ...     {
    ...         '1': [ServerVersion(123, 1, 4000)],
    ...         '2': [ServerVersion(123, 1, 4000)]
    ...     }
    ... )
    True
    >>> base_versions_synchronized(
    ...     {
    ...         ServerVersion(123, 1, 4000): [1],
    ...     },
    ...     {
    ...         '1': [ServerVersion(123, 2, 5000)],
    ...         '2': [ServerVersion(123, 2, 5000)]
    ...     }
    ... )
    False
    >>> base_versions_synchronized(
    ...     {
    ...         ServerVersion(123, 1, 4000): [1],
    ...         ServerVersion(122, 2, 3000): [2],
    ...     },
    ...     {
    ...         '1': [ServerVersion(123, 2, 5000)],
    ...         '2': [ServerVersion(123, 2, 5000)]
    ...     }
    ... )
    False
    >>> base_versions_synchronized(
    ...     {
    ...         ServerVersion(123, 1, 4000): [2, 3],
    ...         ServerVersion(122, 2, 3000): [1],
    ...     },
    ...     {
    ...         '1': [ServerVersion(123, 1, 4000)],
    ...         '2': [ServerVersion(123, 2, 5000)]
    ...     }
    ... )
    False
    >>> base_versions_synchronized(
    ...     {
    ...         ServerVersion(123, 1, 4000): [1],
    ...     },
    ...     {
    ...         '1': [ServerVersion(123, 1, 4000)],
    ...         '2': [ServerVersion(123, 2, 5000)]
    ...     }
    ... )
    True
    >>> base_versions_synchronized(
    ...     {
    ...         ServerVersion(123, 1, 4000): [1],
    ...     },
    ...     {
    ...         '1': [ServerVersion(123, 1, 4000)],
    ...         '2': [ServerVersion(123, 2, 5000)]
    ...     }
    ... )
    True
    """
    cs_base_versions_by_namespace = {namespace: [version.basever for version in versions] for namespace, versions in deployed_cs_versions.items()}
    cs_base_versions = list(set(chain(*(cs_base_versions_by_namespace.values()))))

    server_base_versions = {version.basever: services for version, services in deployed_server_versions.items()}

    return len(cs_base_versions) == 1 and cs_base_versions[0] == max(server_base_versions.items(), key=lambda x: len(x[1]))[0] or (
        len(server_base_versions) == 1 and
        any([
            len(base_versions) == 1 and base_versions[0] == server_base_versions.keys()[0]
            for base_versions in cs_base_versions_by_namespace.values()
        ])
    )


def get_namespace_to_deploy(deployed_server_version, deployed_cs_versions):
    """
    >>> get_namespace_to_deploy(
    ...     ServerVersion(122, 2, 3000),
    ...     {
    ...         '1': [ServerVersion(122, 2, 3000)],
    ...         '2': [ServerVersion(121, 3, 1000)],
    ...     }
    ... )  # with only one CS instance updated to stable full version
    '2'
    >>> get_namespace_to_deploy(
    ...     ServerVersion(122, 2, 3000),
    ...     {
    ...         '1': [ServerVersion(122, 1, 3000)],
    ...         '2': [ServerVersion(121, 3, 1000)],
    ...     }
    ... )  # with only one CS instance updated to stable base version only
    '2'
    >>> get_namespace_to_deploy(
    ...     ServerVersion(122, 2, 3000),
    ...     {
    ...         '1': [ServerVersion(122, 2, 3000)],
    ...         '2': [ServerVersion(122, 2, 3000)],
    ...     }
    ... )  # with both CS instances updated to stable full version
    '2'
    >>> get_namespace_to_deploy(
    ...     ServerVersion(122, 2, 3000),
    ...     {
    ...         '1': [ServerVersion(122, 1, 3000)],
    ...         '2': [ServerVersion(122, 1, 3000)],
    ...     }
    ... )  # with both CS instances updated to stable basever only
    '2'
    >>> get_namespace_to_deploy(
    ...     ServerVersion(122, 2, 3000),
    ...     {
    ...         '1': [ServerVersion(122, 2, 3000)],
    ...         '2': [ServerVersion(121, 1, 1000), (122, 2, 3000)],
    ...     }
    ... )  # with one CS instance is currently updating
    '2'
    >>> get_namespace_to_deploy(
    ...     ServerVersion(122, 1, 3000),
    ...     {
    ...         '1': [ServerVersion(122, 1, 3000)],
    ...         '2': [ServerVersion(123, 1, 2000)],
    ...     }
    ... )  # hotfix without changing basever
    '2'
    >>> get_namespace_to_deploy(
    ...     ServerVersion(122, 1, 3000),
    ...     {
    ...         '1': [ServerVersion(122, 1, 3000)],
    ...         '2': [ServerVersion(123, 1, 2000)],
    ...     }
    ... )  # hotfix with changing basever
    '2'
    >>> get_namespace_to_deploy(
    ...     ServerVersion(122, 1, 3000),
    ...     {
    ...         '1': [ServerVersion(122, 1, 3000)]
    ...     }
    ... )
    Traceback (most recent call last):
        ...
    Exception: Incorrect state: 1 namespaces has production version 122-1.3000, 0 namespaces has non-production versions
    >>> get_namespace_to_deploy(
    ...     ServerVersion(122, 2, 3000),
    ...     {
    ...         '1': [ServerVersion(123, 1, 2000)],
    ...         '2': [ServerVersion(123, 1, 1000)],
    ...     }
    ... )
    Traceback (most recent call last):
        ...
    Exception: Incorrect state: no namespaces with same version as deployed frontend version 122-2.3000 found
    """
    prod_namespaces = []
    candidate_namespaces = []
    for namespace, versions in sorted(deployed_cs_versions.items(), key=lambda x: x[0]):
        if len(versions) == 1 and (deployed_server_version in versions or deployed_server_version.basever == versions[0].basever):
            prod_namespaces.append(namespace)
        else:
            candidate_namespaces.append(namespace)

    if len(prod_namespaces) < 1:
        raise Exception('Incorrect state: no namespaces with same version as deployed frontend version {} found'.format('{}-{}.{}'.format(*deployed_server_version)))
    if len(candidate_namespaces) > 0:
        return candidate_namespaces[0]
    elif len(prod_namespaces) > 1:
        return prod_namespaces[-1]
    else:
        raise Exception('Incorrect state: {} namespaces has production version {}, {} namespaces has non-production versions'.format(
            len(prod_namespaces), '{}-{}.{}'.format(*deployed_server_version), len(candidate_namespaces)))
