import logging
import traceback
import textwrap
from datetime import timedelta, datetime

from dateutil import parser

from sandbox import sdk2

from sandbox.sdk2.vcs.svn import Arcadia
from sandbox.common.errors import TaskFailure
from sandbox.common.utils import get_task_link
from sandbox.projects.release_machine.tasks.GetLastGoodRevision import (
    GetLastGoodRevision,
    NoGoodRevisionFound,
    TimeDelta,
)
from sandbox.projects.release_machine.tasks.GetLastGoodRevision.problems import (
    UnresolvedProblem,
    CheckTestIsNotOK,
    RevisionProblem,
)
from sandbox.projects.yabs.release.duty.schedule import get_current_responsibles
from sandbox.projects.release_machine import rm_notify
from sandbox.projects.release_machine.core import const as rm_const
from sandbox.projects.release_machine.components import all as rmc

logger = logging.getLogger(__name__)


class RevisionsDistance(RevisionProblem):
    description = "Revision was commited too close to the previous release"


class PreviousReleaseTimeDistance(RevisionProblem):
    description = "Revision was commited too soon after the previous release"


class HeadTimeDistance(RevisionProblem):
    description = "Revision was commited too long ago"


def get_commit_datetime(rev):
    arc_url = Arcadia.trunk_url(revision=rev)
    commit_date = Arcadia.info(arc_url)["date"]
    return parser.parse(commit_date).replace(tzinfo=None)


class GetLastGoodRevisionWithRetries(GetLastGoodRevision):
    class Parameters(GetLastGoodRevision.Parameters):
        with sdk2.parameters.Group("Revision distance settings") as distance_settings:
            required_minimal_delay = TimeDelta("Minimal allowed time interval between start revision and desired", description="Interval in seconds or timedelta in HH:MM:SS format", default=0)
            required_max_head_delay = TimeDelta("Maximal allowed time interval between top_limit_revision and desired", description="Interval in seconds or timedelta in HH:MM:SS format", default=0)
            required_minimal_revisions = sdk2.parameters.Integer("Minimal allowed revision interval between start revision and desired", default=0)

        with sdk2.parameters.Group("Retrying") as retrying_parameters:
            wait_period = TimeDelta("Wait period", description="Feel free to use HH:MM:SS format", default="01:00:00")

    def get_tg_recipients(self):
        c_info = rmc.COMPONENTS[self.Parameters.component_name]()
        if hasattr(c_info, "notify_cfg__tg__chats"):
            return [rm_const.RM_USERS[chat].telegram for chat in c_info.notify_cfg__tg__chats]

        return []

    def notify_release_chat(self, message):
        recipients = self.get_tg_recipients()
        logger.debug("Message was: %s", message)
        try:
            rm_notify.send_tm_message(self, message, recipients)
        except Exception:
            logger.error(traceback.format_exc())

    def get_custom_problems(self, revision_candidates):
        custom_problems = {}
        if self.Parameters.required_minimal_revisions > 0:
            last_release_revision = self.start_revision - 1
            for r in revision_candidates:
                if abs(r - last_release_revision) < self.Parameters.required_minimal_revisions:
                    custom_problems.setdefault(r, []).append(RevisionsDistance(r))

        if self.Parameters.required_minimal_delay > 0:
            last_release_revision = self.start_revision - 1
            last_release_revision_time = get_commit_datetime(last_release_revision)
            logger.info("Last release revision date is {}".format(last_release_revision_time))
            for r in revision_candidates:
                r_time = get_commit_datetime(r)
                r_delay = abs(r_time - last_release_revision_time)
                if r_delay.total_seconds() < self.Parameters.required_minimal_delay:
                    custom_problems.setdefault(r, []).append(PreviousReleaseTimeDistance(r))

        if self.Parameters.required_max_head_delay > 0:
            top_limit_time = datetime.now()
            for r in revision_candidates:
                r_time = get_commit_datetime(r)
                r_delay = top_limit_time - r_time
                if r_delay.total_seconds() > self.Parameters.required_max_head_delay:
                    custom_problems.setdefault(r, []).append(HeadTimeDistance(r))

        return custom_problems

    def generate_message(self, c_info, comment=""):
        message_template = textwrap.dedent("""
        <a href=\"{task_link}\">{task_type}</a>: Can not find a green revision for a new release.
        Fix following problems:

        {problems}

        {comment}
        """)
        best_rev, best_rev_problems = max(self.revision_problems.items(), key=lambda x: x[0])

        problem_descriptions = []

        for problem in best_rev_problems:
            if isinstance(problem, UnresolvedProblem):
                try:
                    mention = "@{}".format(rm_notify.get_mention(
                        self,
                        person=problem.owner,
                        component_name=c_info.name,
                    ))
                except Exception:
                    mention = "{}@".format(problem.owner)
            elif not isinstance(problem, (CheckTestIsNotOK, RevisionsDistance, HeadTimeDistance, PreviousReleaseTimeDistance)):
                sandbox_duty_list = get_current_responsibles("SANDBOX")
                sandbox_duty = sandbox_duty_list[0] if sandbox_duty_list else c_info.notify_cfg__st__assignee
                try:
                    mention = "@{}".format(rm_notify.get_mention(
                        self,
                        person=sandbox_duty,
                        component_name=c_info.name,
                    ))
                except Exception:
                    mention = "{}@".format(sandbox_duty)
            else:
                continue

            problem_str = "{}: {}".format(mention, unicode(problem))
            problem_descriptions.append(problem_str)

        return message_template.format(
            task_type=str(self.type),
            task_link=get_task_link(self.id),
            revision=best_rev,
            problems="\n".join(problem_descriptions),
            comment=comment,
        )

    def on_execute(self):
        with self.memoize_stage.first_attempt:
            try:
                super(GetLastGoodRevisionWithRetries, self).on_execute()
            except (TaskFailure, NoGoodRevisionFound):
                c_info = rmc.COMPONENTS[self.Parameters.component_name]()
                message = self.generate_message(
                    c_info,
                    comment="The next attempt to automatically create a release will be performed in {time_delta}.".format(
                        time_delta=timedelta(seconds=self.Parameters.wait_period)
                    ),
                )
                self.notify_release_chat(message)
                raise sdk2.WaitTime(self.Parameters.wait_period)
            except Exception as exc:
                logger.exception(exc)
                raise sdk2.WaitTime(self.Parameters.wait_period)
            else:
                return

        try:
            super(GetLastGoodRevisionWithRetries, self).on_execute()
        except (TaskFailure, NoGoodRevisionFound):
            c_info = rmc.COMPONENTS[self.Parameters.component_name]()
            wiki_link = "https://wiki.yandex-team.ru/bannernajakrutilka/server/relizy/releasecyclev20/#prereleaseavtomaticheskaja/ruchnajaoperacija"
            message = self.generate_message(c_info, comment="I'm tired. <a href=\"{wiki_link}\">Create a new release manually</a> after resolving issues above.".format(wiki_link=wiki_link))
            self.notify_release_chat(message)
            raise
