import logging
import json
import itertools
from datetime import datetime

from sandbox import sdk2

from sandbox.common.types.notification import Transport
from sandbox.projects.release_machine import input_params2 as rm_params
from sandbox.projects.release_machine.core import const as rm_const, task_env
from sandbox.projects.release_machine.components.configs.yabs_server import YabsServerCfg
from sandbox.projects.release_machine.components.all import get_component
from sandbox.projects.release_machine import resources as rm_res

from .message import create_message_body, SUBJECT_TEMPLATE


class YabsServerStableReleaseNotify(sdk2.Task):
    """Sends notifications about new stable release
    """
    class Requirements(sdk2.Requirements):
        cores = 1
        ram = 4096

        environments = [task_env.TaskRequirements.startrek_client]
        client_tags = task_env.TaskTags.all_rm & task_env.TaskTags.startrek_client

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Parameters):
        changelog = sdk2.parameters.Resource("Full changelog resource", resource_type=rm_res.RELEASE_MACHINE_CHANGELOG)
        component_name = rm_params.ComponentName2.component_name(default=YabsServerCfg.name)
        release_number = sdk2.parameters.Integer("Release number", required=True)  # TODO: remove
        revision = sdk2.parameters.Integer("Last revision", required=True)  # TODO: remove
        recipients = sdk2.parameters.List("Logins to send message to", default=["yabs-dev"])
        filter_finalized_issues_by_statuses = sdk2.parameters.List("Filter finalized issues by statuses", default=["rc", "awaitingDeployment", "deployed"])
        filter_wip_issues_by_statuses = sdk2.parameters.List("Filter WIP issues by statuses", default=["inProgress", "needInfo"])
        st_token = sdk2.parameters.String("Startrek token vault name", default_value=rm_const.COMMON_TOKEN_NAME)

    @property
    def startrek_token(self):
        if not hasattr(self, "__startrek_token"):
            self.__startrek_token = sdk2.Vault.data(self.Parameters.st_token)
        return self.__startrek_token

    @staticmethod
    def get_changes(changelog_resource):
        resource_data = sdk2.ResourceData(changelog_resource)
        resource_path = str(resource_data.path)
        with open(resource_path, "r") as changelog_file:
            changelog = json.load(changelog_file)
        return list(
            itertools.chain(
                *(changes.get("changes", []) for changes in changelog.get("all_changes", []))
            )
        )

    @staticmethod
    def get_important_change_issues(changes, component_info, startrek_client, filter_finalized_issues_by_statuses, filter_wip_issues_by_statuses):
        finalized_issues = set()
        wip_issues = set()
        for c in changes:
            if c["commit_importance"] >= component_info.notify_cfg__st__commit_importance_threshold:
                for issue_key in c.get("startrek_tickets", ()):
                    try:
                        issue = startrek_client.issues[issue_key]
                    except Exception:
                        logging.exception("Failed to get issue %s", issue_key)
                    else:
                        if issue.status.key in filter_finalized_issues_by_statuses:
                            finalized_issues.add((issue_key, issue.summary))
                        elif issue.status.key in filter_wip_issues_by_statuses:
                            wip_issues.add((issue_key, issue.summary))
                        else:
                            logging.debug("Issue %s has status %s and is not included in release notes", issue_key, issue.status.key)

        return finalized_issues, wip_issues

    def send_message(self, finalized_issues, wip_issues, major_version, minor_version, recipients):
        message = create_message_body(finalized_issues, wip_issues, datetime.today(), major_version, minor_version)
        subject = SUBJECT_TEMPLATE.format(major_version=str(major_version), minor_version=str(minor_version))
        logging.debug("Subject: %s", subject)
        logging.debug("Message body: %s", message)
        self.server.notification(
            subject=subject,
            body=message,
            recipients=recipients,
            transport=Transport.EMAIL,
            type="html",
        )

    def on_execute(self):
        changes = self.get_changes(self.Parameters.changelog)
        component_info = get_component(self.Parameters.component_name)

        from startrek_client import Startrek
        startrek_client = Startrek(useragent=self.__class__.__name__, token=self.startrek_token)

        finalized_issues, wip_issues = self.get_important_change_issues(
            changes,
            component_info,
            startrek_client,
            self.Parameters.filter_finalized_issues_by_statuses,
            self.Parameters.filter_wip_issues_by_statuses,
        )

        self.send_message(finalized_issues, wip_issues, self.Parameters.changelog.major_release_num, self.Parameters.changelog.minor_release_num, self.Parameters.recipients)
