import logging
import os

from sandbox import sdk2
from sandbox.common.utils import get_resource_link
from sandbox.projects.abc.client import AbcClient
from sandbox.projects.release_machine import input_params2 as rm_params
from sandbox.projects.release_machine import security as rm_sec
from sandbox.projects.release_machine.components.all import get_component
from sandbox.projects.release_machine.components.components_info import Branched
from sandbox.projects.release_machine.core import task_env
from sandbox.projects.release_machine.helpers.startrek_helper import STHelper
from sandbox.projects.yabs.qa.template_utils import get_template
from sandbox.projects.yabs.release.notifications.environment.report_info import TaskInfo
from sandbox.projects.yabs.release.notifications.jns.client import send_message
from sandbox.projects.yabs.release.notifications.jns.helpers import get_logins

logger = logging.getLogger(__name__)


TEMPLATES_DIR = os.path.join(os.path.dirname(__file__), '..', '..', 'notifications', 'templates')


class TagInfo(object):
    __slots__ = ('text', 'url')
    __url_template = 'https://a.yandex-team.ru/{c_info.svn_cfg__REPO_NAME}/{c_info.svn_cfg__tag_folder}/{c_info.svn_cfg__tag_name}/{tag}'

    def __init__(self, component_name, major_version, minor_version):
        self.text = 'r{}-{}'.format(major_version, minor_version)
        self.url = self.__url_template.format(c_info=get_component(component_name), tag=self.text)


class DeployLink(object):
    __slots__ = ('text', 'url')
    __text_template = "Deploy {component_name} r{major_version}-{minor_version}"
    __url_template = "https://rm.z.yandex-team.ru/component/{component_name}/manage?branch={major_version}&tag={minor_version}"

    def __init__(self, component_name, major_version, minor_version):
        self.text = self.__text_template.format(component_name=component_name, major_version=major_version, minor_version=minor_version)
        self.url = self.__url_template.format(component_name=component_name, major_version=major_version, minor_version=minor_version)


class ResourceInfo(object):
    __slots__ = ('type', 'id', 'url')

    def __init__(self, resource_type, resource_id):
        self.type = resource_type
        self.id = resource_id
        self.url = get_resource_link(resource_id)


class ReportData(object):
    __slots__ = ('task', 'component_name', 'artifacts', 'deploy_links', 'tag', 'tags')

    def __init__(
            self,
            task_id=0,
            task_type='YABS_SERVER_REPORT_BUILD',
            component_name='yabs_server',
            major_version=0,
            minor_version=0,
            artifacts=None,
    ):
        self.task = TaskInfo(task_type, task_id) if task_id else None
        self.component_name = component_name
        self.artifacts = artifacts or {}
        self.tags = ['build', 'r{}'.format(major_version)] + (['r{}_{}'.format(major_version, minor_version)] if minor_version else [])
        self.tag = TagInfo(component_name=self.component_name, major_version=major_version, minor_version=minor_version) if minor_version else None
        self.deploy_links = [
            DeployLink(component_name=self.component_name, major_version=major_version, minor_version=minor_version),
        ] if major_version and minor_version else []

    def as_dict(self, transport='telegram', mentions=()):
        _dict = {k: getattr(self, k) for k in self.__slots__}
        _dict.update(transport=transport, mentions=mentions)
        return _dict


class YabsServerReportBuild(sdk2.Task):
    """Report build artifacts to st and tg
    """

    __st_helper = None
    __component_info = None

    class Parameters(rm_params.ComponentNameResources):
        release_number = sdk2.parameters.Integer("Release number")

        with sdk2.parameters.Group("Report settings") as report_settings:
            post_st_comment = sdk2.parameters.Bool("Post build info to startrek release ticket", default=True)
            send_tg_message = sdk2.parameters.Bool("Send build info to component-defined telegram chat", default=False)

    class Requirements(sdk2.Requirements):
        cores = 1  # exactly 1 core
        ram = 4096  # 4GiB or less

        environments = [task_env.TaskRequirements.startrek_client]
        client_tags = task_env.TaskTags.all_rm & task_env.TaskTags.startrek_client

        class Caches(sdk2.Requirements.Caches):
            pass

    @property
    def st_helper(self):
        if self.__st_helper is None:
            token = rm_sec.get_rm_token(self)
            self.__st_helper = STHelper(token)
        return self.__st_helper

    @property
    def component_info(self):
        if self.__component_info is None:
            self.__component_info = get_component(self.Parameters.component_name)
        return self.__component_info

    def get_report_data(self):
        major_version, minor_version = self.get_tag_data()
        return ReportData(
            component_name=self.Parameters.component_name,
            major_version=major_version,
            minor_version=minor_version,
            task_id=self.id,
            task_type=str(self.type),
            artifacts=self.get_artifacts(),
        )

    def get_tag_data(self):
        for resource_info in self.component_info.releases_cfg__resources_info:
            if resource_info.resource_name not in self.Parameters.component_resources:
                continue
            resource = sdk2.Resource[self.Parameters.component_resources[resource_info.resource_name]]

            if isinstance(self.component_info, Branched):
                resource_branch_number, resource_tag_number = self.component_info.get_tag_info_from_build_task(resource.task.id, resource_info.build_ctx_key)
                if resource_branch_number is not None:
                    branch_number, tag_number = resource_branch_number, resource_tag_number
                    return branch_number, tag_number

        return self.Parameters.release_number, None

    def get_artifacts(self):
        artifacts = {}
        for resource_name, resource_id in self.Parameters.component_resources.items():
            resource = sdk2.Resource[resource_id]
            artifacts.setdefault(TaskInfo(resource.task.type, resource.task.id), []).append(ResourceInfo(str(resource.type), str(resource.id)))
        return artifacts

    def generate_report(self, report_data, spawn_users, template_name='build_report.j2', transport='telegram'):
        mentions = get_logins(self, spawn_users, transport=transport)
        report_template_j2 = get_template(template_name, templates_dir=TEMPLATES_DIR)
        return report_template_j2.render(report_data.as_dict(transport=transport, mentions=mentions))

    def get_recipients(self):
        if self.Context.copy_of:
            return {
                "yachats": {
                    "internal": [{"login": self.author}],
                },
                "telegram": {
                    "internal": [{"login": self.author}],
                },
            }
        return {
            "yachats": {
                "chat_name": ["yabs_server_release_chat"]
            },
            "telegram": {
                "chat_name": ["yabs_server_release_chat"]
            },
        }

    def send_st_report(self, release_number, comment):
        self.st_helper.comment(release_number, comment, self.component_info)

    def on_execute(self):
        report_data = self.get_report_data()
        tokens = sdk2.yav.Secret("sec-01fx7jcsjevejnypw63tk26nj3").data()
        spawn_users = [AbcClient(tokens['abc_token']).get_current_duty_login(179, schedule_slug='yabs_frontend_duty_first')]

        if self.Parameters.post_st_comment:
            st_report = self.generate_report(report_data, spawn_users=spawn_users, transport='startrek')
            logger.debug(st_report)
            self.send_st_report(self.Parameters.release_number, st_report)

        if self.Parameters.send_tg_message:
            for transport, recipient in self.get_recipients().items():
                report = self.generate_report(report_data, spawn_users=spawn_users, transport=transport)
                send_message(report, tokens['juggler_token'], recipients={transport: recipient})
