# -*- coding: utf-8 -*-
from collections import defaultdict, namedtuple
import logging
from datetime import datetime, timedelta
import os

from sandbox import sdk2
from sandbox.common.utils import get_task_link

from sandbox.projects.common.testenv_client.api_client import TestenvApiClient

from sandbox.projects.yabs.release.binary_search import intervals, report, testenv
from sandbox.projects.yabs.release.common import BaseReleaseTask

from sandbox.projects.yabs.qa.template_utils import get_template
from sandbox.projects.yabs.release.notifications.jns.helpers import get_logins
from sandbox.projects.yabs.release.notifications.jns.client import send_message_to_channel
from sandbox.projects.abc.client import AbcClient


YAV_SECRET = "sec-01fx7jcsjevejnypw63tk26nj3"


TESTS = [
    "YABS_SERVER_40_FT_BS_A_B",
    "YABS_SERVER_40_FT_BSRANK_A_B",
    "YABS_SERVER_40_FT_YABS_A_B",
    "YABS_SERVER_40_PERFORMANCE_BEST_BS_SAMPLED",
    "YABS_SERVER_40_PERFORMANCE_BEST_BSRANK_SAMPLED",
    "YABS_SERVER_40_PERFORMANCE_BEST_YABS_SAMPLED",
    "YABS_SERVER_45_PERFORMANCE_META_BS_A_B_SAMPLED",
    "YABS_SERVER_45_PERFORMANCE_META_BSRANK_A_B_SAMPLED",
    "YABS_SERVER_45_PERFORMANCE_META_YABS_A_B_SAMPLED"
]

WAIT_TIME = 1200  # 20 minutes
WAIT_TIMEOUT = timedelta(hours=2, minutes=30)


CHAT_NAME = "yabs_server_release_chat"
RECIPIENTS = {"telegram": {"chat_name": [CHAT_NAME]}, "yachats": {"chat_name": [CHAT_NAME]}}
NOTIFICATION_TEMPLATE_NAME = 'unresolved_diffs_status.j2'

DiffKey = namedtuple('DiffKey', 'staff telegram')
DiffValue = namedtuple('DiffValue', 'interval_end database')


class YabsServerGetB2BDiffs(BaseReleaseTask):
    name = 'YABS_SERVER_GET_B2B_DIFFS'

    class Parameters(BaseReleaseTask.Parameters):
        test_names = sdk2.parameters.List("List of diff tests to report", default=TESTS)

        release_confirm_people = sdk2.parameters.List("People, who can confirm releases (logins)")

        notification_is_enabled = sdk2.parameters.Bool("Notify about Diffs", default=False)
        wait_on_diffs = sdk2.parameters.Bool("Wait when there are Diffs", default=False)

    class Context(sdk2.Context):
        tests_with_diffs = list()
        next_notify_time = None

    @staticmethod
    def get_problems(testenv_client, database_name, test_names):
        problems = testenv_client.get_te_problems(database_name, unresolved_only=False)["rows"]
        return {
            test_name: {
                problem["test_diff/revision2"]: problem
                for problem in problems
                if problem["test_name"] == test_name
            } for test_name in test_names
        }

    def on_create(self):
        self.Context.next_notify_time = (datetime.now() + WAIT_TIMEOUT).strftime("%Y-%m-%dT%H:%M")

    def on_execute(self):
        release_ticket = self.__get_release_ticket()

        if not self.check_release_ticket(release_ticket):
            return

        begin = self.get_start_revision()
        end = self.get_final_revision()

        tests_to_execute = self.Parameters.test_names
        if self.Context.tests_with_diffs:
            tests_to_execute = self.Context.tests_with_diffs
            self.Context.tests_with_diffs = list()

        tests_results = self.__get_tests_results(self.Parameters.database_name, tests_to_execute, begin, end)

        self.create_comments_with_test_results(tests_results, release_ticket, tests_to_execute)

        self.notify_about_results(tests_results, release_ticket)

    def check_release_ticket(self, release_ticket):
        if release_ticket.status.key == 'readyToDeploy':
            self.set_info('Release ticket {} is already in Ready To Deploy status, terminating.'.format(self.__get_release_ticket_ref(release_ticket)))
            return False
        else:
            return True

    def create_comments_with_test_results(self, tests_results, release_ticket, tests_to_execute):
        previous_test_result_comments = self.__get_test_comments(release_ticket, tests_to_execute)

        for test_name in tests_results:
            interval_sequence, problems_by_interval = tests_results[test_name]
            summary_rows, interval_table_rows = report.get_report_data(self.server, interval_sequence, problems_by_interval)
            comment = report.create_startrek_report(summary_rows, interval_table_rows, test_name)

            comment += '\n\nSent by (({task_link} {task_id}))'.format(task_link=get_task_link(self.id), task_id=self.id)

            if test_name in previous_test_result_comments:
                previous_test_result_comments[test_name].update(text=comment)
            else:
                release_ticket.comments.create(text=comment)

    def notify_about_results(self, tests_results, release_ticket):
        unresolved_diffs_by_user = defaultdict(list)
        unresolved_diffs_without_problems = dict()
        tests_with_diffs = set()

        for test_name in tests_results:
            interval_sequence, problems_by_interval = tests_results[test_name]

            for interval in interval_sequence:
                if interval.status == intervals.NO_DIFF or interval.status == intervals.FIXED:
                    continue

                problem = problems_by_interval.get(interval.end, {})

                owner = interval.owner
                if not owner:
                    owner = release_ticket.assignee

                # In this case add CMP Task link for the interval
                if not problem:
                    logging.error('Problem not found for Interval {}. Owner: {} Database: {}'.format(interval.end, interval.owner, interval.database))

                    tests_with_diffs.add(test_name)
                    # unresolved_diffs_by_user[owner].append(DiffValue(interval.end, interval.database, 'Problem by Interval is not found'))
                    if interval.end not in unresolved_diffs_without_problems:
                        unresolved_diffs_without_problems[interval.end] = DiffValue(interval.end, interval.database)
                    continue

                if problem.get("status", "") != "resolved":
                    unresolved_diffs_by_user[owner].append(DiffValue(interval.end, interval.database))
                    tests_with_diffs.add(test_name)

        if tests_with_diffs:
            self.Context.tests_with_diffs = list(tests_with_diffs)

            if (datetime.now()).strftime("%Y-%m-%dT%H:%M") > self.Context.next_notify_time:
                logging.info('Notify timeout {}'.format(self.Context.next_notify_time))
                self.__notify_about_unresolved_diffs(release_ticket, unresolved_diffs_by_user, unresolved_diffs_without_problems)

                self.Context.next_notify_time = (datetime.now() + WAIT_TIMEOUT).strftime("%Y-%m-%dT%H:%M")

            with self.memoize_stage.notify_once:
                self.__notify_about_unresolved_diffs(release_ticket, unresolved_diffs_by_user, unresolved_diffs_without_problems)

            self.__wait()
        else:
            self.__notify_about_success(release_ticket)

    def __get_release_ticket(self):
        release_ticket = self.get_release_ticket()
        st_client = self.__get_st_client(self.__get_token_from_vault(self.Parameters.st_vault_name))

        return st_client.issues[release_ticket]

    def __get_release_ticket_ref(self, release_ticket):
        return '<a href="https://st.yandex-team.ru/{issue_key}" target="_blank">{issue_key}</a>'.format(issue_key=release_ticket.key)

    def __get_token_from_vault(self, vault_name):
        return sdk2.Vault.data(vault_name)

    def __get_st_client(self, st_token):
        from startrek_client import Startrek
        return Startrek(useragent='YABS_SERVER_GET_B2B_DIFFS', base_url='https://st-api.yandex-team.ru', token=st_token)

    def __get_testenv_client(self):
        testenv_token = self.__get_token_from_vault(self.Parameters.te_vault_name)
        return TestenvApiClient(token=testenv_token)

    def __get_tests_results(self, database_name, test_names, begin, end):
        results = dict()
        logging.debug('__get_tests_results')

        for test in test_names:
            interval_map, problems_by_interval = testenv.get_interval_info_for_test(self.__get_testenv_client(), database_name, test, begin, end)
            interval_sequence = intervals.Partitioner(interval_map).partition_interval((begin, end))

            results[test] = (interval_sequence, problems_by_interval)
            logging.debug('Test results for {}'.format(test))
            logging.debug(interval_sequence)
            logging.debug(problems_by_interval)

        return results

    def __get_test_comments(self, release_ticket, test_list):
        logging.debug('__get_test_comments')
        result = dict()

        comments = release_ticket.comments.get_all()
        for i, comment in enumerate(comments):
            for test_name in test_list:
                search_pattern = '==== {} ===='.format(test_name)
                if search_pattern in comment.text:
                    result[test_name] = comment
                    logging.debug('{}. {} {}'.format(i, test_name, comment))

        return result

    def __wait(self):
        if self.Parameters.wait_on_diffs:
            raise sdk2.WaitTime(WAIT_TIME)

    def __get_juggler_token(self):
        return self.__get_token('juggler_token')

    def __get_abc_token(self):
        return self.__get_token('abc_token')

    def __get_token(self, token_name):
        tokens = sdk2.yav.Secret(YAV_SECRET).data()
        return tokens[token_name]

    def __notify_about_success(self, release_ticket):
        self.__notify(release_ticket, 0, [], [])

    def __notify_about_unresolved_diffs(self, release_ticket, unresolved_diffs_by_user, diffs_without_problems):
        diff_count = 0
        diffs = defaultdict(list)

        for user_name, revisions in unresolved_diffs_by_user.items():
            telegram_user = get_logins(self, [user_name])
            if telegram_user:
                telegram_user = telegram_user[0]
            else:
                telegram_user = user_name

            used_revisions = set()
            for revision in revisions:
                if revision.interval_end not in used_revisions:
                    diffs[DiffKey(user_name, telegram_user)].append(revision)
                    used_revisions.add(revision.interval_end)
                    diff_count += 1

        diff_count += len(diffs_without_problems)
        self.__notify(release_ticket, diff_count, diffs, list(diffs_without_problems.values()))

    def __get_on_duty_users(self):
        on_duty_users = []
        try:
            on_duty_users.append(AbcClient(self.__get_abc_token()).get_current_duty_login(179, schedule_slug='yabs_frontend_duty_first'))
        except Exception:
            self.set_info('Cannot get On Duty Users')
            logging.error('Cannot get On Duty Users', exc_info=True)

        logging.debug("On Duty Users: {}".format(' '.join(on_duty_users)))
        return on_duty_users

    def __get_notification_template(self):
        template_dir = os.path.join(os.path.dirname(__file__), '..', '..', 'notifications', 'templates')
        try:
            # Non binary task is executed here "/home/zomb-sandbox/tasks/projects/yabs/release/tasks/YabsServerGetB2BDiffs/__init__.py"
            return get_template(NOTIFICATION_TEMPLATE_NAME, templates_dir=template_dir)
        except Exception:
            logging.error('Cannot find Notification Template in {}'.format(template_dir), exc_info=True)

            # In case of local build
            template_dir = 'sandbox/projects/yabs/release/notifications/templates/'
            logging.info('Try to load template in {}'.format(template_dir))
            return get_template(NOTIFICATION_TEMPLATE_NAME, templates_dir=template_dir)

    def __notify(self, release_ticket, diff_count, diffs, diffs_without_problems):
        logging.debug('Diffs count: {}'.format(diff_count))
        logging.debug(diffs)
        logging.debug(diffs_without_problems)

        release_confirm_users = self.Parameters.release_confirm_people
        if not release_confirm_users or diff_count == 0:
            release_confirm_users = self.__get_on_duty_users()
        logging.debug("Release confirm users: {}".format(' '.join(release_confirm_users)))

        if not self.Parameters.notification_is_enabled:
            logging.info("Notification is disabled in Sandbox Task parameters")
            return

        now = datetime.utcnow() + timedelta(hours=3)  # Moscow Time
        # Mon: 0, Tue: 1, Wed: 2, Thu: 3, Fri: 4, Sat: 5, Sun: 6
        if now.weekday() > 4 or now.hour > 19 or now.hour < 8:
            logging.info("Notification is enabled only during working hours")
            return

        notification_template = self.__get_notification_template()

        for transport, recipient in RECIPIENTS.items():
            mentions = get_logins(self, release_confirm_users, transport=transport)

            try:
                message = self.__get_notification_message(notification_template, transport, release_ticket, diff_count, diffs, diffs_without_problems, mentions)

                logging.debug(message)
                send_message_to_channel(message, self.__get_juggler_token(), channel=transport)
            except Exception:
                self.set_info('Cannot send notification')
                logging.error('Cannot send notification', exc_info=True)

    def __get_notification_message(self, notification_template, transport, release_ticket, diff_count, diffs, diffs_without_problems, mentions):
        return notification_template.render({'release_ticket': release_ticket.key,
                                             'release_ticket_link': "http://st.yandex-team.ru/{}".format(release_ticket.key),
                                             'transport': transport,
                                             'diff_count': diff_count,
                                             'on_duty_users': mentions,
                                             'diffs': diffs,
                                             'diffs_without_problems': diffs_without_problems})
