import logging
import posixpath

from sandbox import sdk2

from sandbox.projects.release_machine.helpers.startrek_helper import STHelper
from sandbox.projects.release_machine.input_params2 import ComponentName2
from sandbox.projects.release_machine.components import all as rmc
from sandbox.projects.release_machine import security as rm_sec
from sandbox.projects.release_machine.core import task_env
from sandbox.projects.yabs.qa.tasks.YabsServerCheckFT.ci import get_ci_check_url_ui, TEST_TYPES
from sandbox.projects.yabs.qa.tasks.YabsServerCheckFT import YABS_SERVER_TEST_PATH


logger = logging.getLogger(__name__)


class YabsServerGetFTResults(sdk2.Task):
    name = "YABS_SERVER_GET_FT_RESULTS"

    __st_helper = None
    __component_info = None

    class Parameters(ComponentName2):
        release_number = sdk2.parameters.Integer("Release number")

        with sdk2.parameters.Group('Report parameters') as report_parameters:
            test_paths = sdk2.parameters.List("Test paths", default=[YABS_SERVER_TEST_PATH])
            with sdk2.parameters.CheckGroup("Test types") as test_types:
                for test_type in TEST_TYPES:
                    test_types.values[test_type] = test_types.Value(test_type, checked=True)

    class Requirements(sdk2.Requirements):
        cores = 1  # exactly 1 core
        ram = 4096  # 4GiB or less

        environments = [task_env.TaskRequirements.startrek_client]
        client_tags = task_env.TaskTags.all_rm & task_env.TaskTags.startrek_client

        class Caches(sdk2.Requirements.Caches):
            pass

    @property
    def st_helper(self):
        if self.__st_helper is None:
            token = rm_sec.get_rm_token(self)
            self.__st_helper = STHelper(token)
        return self.__st_helper

    @property
    def component_info(self):
        if self.__component_info is None:
            self.__component_info = rmc.COMPONENTS[self.Parameters.component_name]()
        return self.__component_info

    def create_st_report(self, test_paths, test_types, release_number):
        branch_name = posixpath.join(
            self.component_info.svn_cfg__branch_name,
            self.component_info.svn_cfg__branch_folder_name(release_number)
        )
        return "FT results are available in (({} CI)).".format(get_ci_check_url_ui(test_paths, test_types=test_types, branch_name=branch_name))

    def send_st_report(self, release_number, comment):
        self.st_helper.comment(release_number, comment, self.component_info)

    def on_execute(self):
        st_report = self.create_st_report(self.Parameters.test_paths, self.Parameters.test_types, self.Parameters.release_number)
        logger.debug(st_report)
        self.send_st_report(self.Parameters.release_number, st_report)
