# Parameters.release_ticket, -*- coding: utf-8 -*-
# pylint: disable=C0301

import logging
import re
import requests
import urlparse
from collections import defaultdict

from sandbox import sdk2
from sandbox.projects.common.yabs.server.util import send_startrek_report

from sandbox.projects.yabs.release.common import BaseReleaseTask


TESTENV_API_BASE_URL = 'https://testenv.yandex-team.ru/handlers/'
TESTENV_API_TEST_RESULTS = urlparse.urljoin(TESTENV_API_BASE_URL, 'grids/testResults')
TESTENV_API_DIFF_TASK_ID = urlparse.urljoin(TESTENV_API_BASE_URL, 'getTestDiff')
TESTENV_API_PROBLEMS_INFO = urlparse.urljoin(TESTENV_API_BASE_URL, 'grids/problems')
TEST_NAME_PATTERN = 'YABS_SERVER_40_PERFORMANCE_BEST_{mode}'
ARCADIA_COMMIT_LINK = '((https://a.yandex-team.ru/arc/commit/{revision} {revision}))'
CA_CERT_PATH = '/etc/ssl/certs/YandexInternalRootCA.pem'
SANDBOX_TASK_LINK = '((https://sandbox.yandex-team.ru/task/{task_id}/view {task_id}))'

MODES = ['YABS', 'BS', 'BSRANK']

RPS_RESULT_PATTERN = re.compile('Max requests/sec changed by [^\n]+')

HTTP_TIMEOUT = 60

OK_COLOR = 'green'
FAIL_COLOR = 'red'

NO_DIFF = 'No diff'
RESOLVED_DIFF = 'Resolved diff'
NOT_RESOLVED_DIFF = 'Not resolved diff'
DIFF_WITHOUT_PROBLEM = 'Diff without problem'
NO_CMP_TASK = 'No cmp task'

START = 0
LIMIT_PROBLEMS = 100000
LIMIT_TESTS = 1000


def set_session():
    session = requests.Session()
    session.verify = CA_CERT_PATH
    return session


class YabsServerGetPerformanceDiffs(BaseReleaseTask):

    class Parameters(BaseReleaseTask.Parameters):
        pass

    def get_problems_data(self, session):
        params = {
            'database': self.Parameters.database_name,
            'limit': LIMIT_PROBLEMS,
            'start': START
        }
        return session.get(TESTENV_API_PROBLEMS_INFO, params=params, timeout=HTTP_TIMEOUT).json()['rows']

    def get_test_results(self, session, mode):
        params = {
            'database': self.Parameters.database_name,
            'test_name': TEST_NAME_PATTERN.format(mode=mode),
            'hide_filtered': 'true',
            'limit': LIMIT_TESTS,
            'start': START
        }
        return session.get(TESTENV_API_TEST_RESULTS, params=params, timeout=HTTP_TIMEOUT).json()['rows']

    def get_needed_revisions(self, results):
        start_revision = self.get_start_revision()
        final_revision = self.get_final_revision()
        needed_revisions = list(set(result['revision'] for result in results if start_revision <= int(result['revision']) <= final_revision and result['status'] == 'OK'))
        return sorted(needed_revisions)

    def get_diff_data(self, session, mode, needed_revisions, problems_data):
        def get_diff_task_id(revision1, revision2):
            params = {
                'database': self.Parameters.database_name,
                'test_name': TEST_NAME_PATTERN.format(mode=mode),
                'revision1': revision1,
                'revision2': revision2
            }
            try:
                return 'testenv', int(session.get(TESTENV_API_DIFF_TASK_ID, params=params, timeout=HTTP_TIMEOUT).json())
            except Exception, e:
                logging.warn("No cmp task in testenv between revisions %s and %s: %s", revision1, revision2, e.message)
                return 'testenv', None

        def get_sandbox_diff_info(diff_task_id):
            info = sdk2.Task[diff_task_id].info
            try:
                return '{} for task_id {}'.format(re.search(RPS_RESULT_PATTERN, info).group(0), SANDBOX_TASK_LINK.format(task_id=diff_task_id))
            except AttributeError:
                logging.warn("There is no necessary data in sandbox task info: task_id=%s", diff_task_id)
                return None

        def get_resolve_status(diff_task_id):
            problem = filter(lambda x: x['test_diff/task_id'] == diff_task_id, problems_data)
            if problem:
                problem = problem[0]
                key = RESOLVED_DIFF if problem['is_resolved'] else NOT_RESOLVED_DIFF
                return {
                    'resolve_comment': problem['resolve_comment'] if problem['resolve_comment'] else ' ',
                    'is_resolved': problem['is_resolved'],
                    'revision2_comment': problem['test_diff/revision2_info/comment'] if problem['test_diff/revision2_info/comment'] else ' ',
                    'status_color': OK_COLOR if problem['is_resolved'] == 'yes' else FAIL_COLOR
                }, key
            return {'status_color': FAIL_COLOR}, DIFF_WITHOUT_PROBLEM

        release_diffs = []
        release_metadata = defaultdict(list)
        for prev_revision, current_revision in zip(needed_revisions, needed_revisions[1:]):
            current_revision_link = ARCADIA_COMMIT_LINK.format(revision=current_revision)
            diff_data = dict(revisions_interval='%s - %s' % (prev_revision, current_revision_link))
            diff_data['status_color'] = OK_COLOR
            diff_data.update(dict.fromkeys(['resolve_comment', 'is_resolved', 'revision2_comment'], ' '))
            source, diff_task_id = get_diff_task_id(prev_revision, current_revision)
            logging.debug(diff_task_id)

            if diff_task_id is not None:
                info = get_sandbox_diff_info(diff_task_id)
                if info is None:
                    info = 'No necessary data in task %s' % diff_task_id
                    diff_data['status_color'] = FAIL_COLOR
                    release_metadata[NO_CMP_TASK].append(current_revision_link)
                else:
                    has_diff = sdk2.Task[diff_task_id].Context.has_diff
                    if has_diff:
                        resolve_data, key = get_resolve_status(diff_task_id)
                        diff_data.update(resolve_data)
                        release_metadata[key].append(current_revision_link)
                    else:
                        release_metadata[NO_DIFF].append(current_revision_link)
            else:
                info = 'No cmp task!!!'
                diff_data['status_color'] = FAIL_COLOR
                release_metadata[NO_CMP_TASK].append(current_revision_link)

            diff_data['sandbox_diff_info'] = info
            logging.debug("sandbox info %s", diff_data['sandbox_diff_info'])

            release_diffs.append(diff_data)
        return release_diffs, release_metadata

    def on_execute(self):
        session = set_session()
        problems_data = self.get_problems_data(session)
        for mode in MODES:
            logging.debug(mode)
            test_results = self.get_test_results(session, mode)
            needed_revisions = self.get_needed_revisions(test_results)
            logging.debug(needed_revisions)
            release_diffs, release_metadata = self.get_diff_data(session, mode, needed_revisions, problems_data)
            wiki_data = create_wiki_data(release_diffs, release_metadata)
            logging.debug(wiki_data)
            token = sdk2.Vault.data(self.Parameters.st_vault_name)
            release_ticket = self.get_release_ticket()
            send_startrek_report(token, release_ticket, TEST_NAME_PATTERN.format(mode=mode) + '\n' + wiki_data, self.id)


# ####### CREATE WIKI TABLE #########

def create_wiki_hyperlinks(diff_info):
    diff_info = re.sub('<a href="', '((', diff_info)
    diff_info = re.sub('</a>', '))', diff_info)
    diff_info = re.sub('" target="_blank">', ' ', diff_info)
    diff_info = re.sub('<div class="hr">[0-9 -:]*</div>', '', diff_info)
    return diff_info


def _color_format(x):
    return '!!({color}){status}!!'.format(color=OK_COLOR if x in [NO_DIFF, RESOLVED_DIFF] else FAIL_COLOR, status=x)


def create_metadata_table(release_metadata):
    wiki_table = '#|\n||**Status**|**Number of intervals**|**Revisions**||\n'
    for status, revisions in release_metadata.items():
        wiki_table += '||' + ' | '.join([_color_format(status), str(len(revisions)), str(revisions) if status != NO_DIFF else '']) + '||\n'
    wiki_table += '|#'
    return wiki_table


def create_diffs_table(release_diffs):
    wiki_table = '#|\n||**Revisions interval**|**Diff info**|**Is resolved**|**Resolve comment**|**Revision2 comment**||\n'
    for diff in release_diffs:
        diff['sandbox_diff_info'] = create_wiki_hyperlinks(diff['sandbox_diff_info'])
        diff['revisions_interval'] = '!!({color}){interval}!!'.format(color=diff['status_color'], interval=diff['revisions_interval'])
        wiki_table += '||' + ' | '.join([diff['revisions_interval'], diff['sandbox_diff_info'], diff['is_resolved'], diff['resolve_comment'], diff['revision2_comment']]) + '||\n'
    wiki_table += '|#'
    return wiki_table


def create_wiki_data(release_diffs, release_metadata):
    metadata_table = create_metadata_table(release_metadata)
    diffs_table = create_diffs_table(release_diffs)
    return '\n'.join([metadata_table, '<{Table of intervals', diffs_table, '}>'])
