import logging
import os
from datetime import datetime, timedelta

from sandbox import sdk2
from sandbox.projects.yabs.qa.template_utils import get_template
from sandbox.projects.yabs.release.notifications.jns.helpers import get_logins
from sandbox.projects.yabs.release.notifications.jns.client import send_message
from sandbox.projects.abc.client import AbcClient
from sandbox.projects.yabs.release.common import BaseTask


YAV_SECRET = "sec-01fx7jcsjevejnypw63tk26nj3"


WAIT_TIME = 600  # 10 minutes
WAIT_TIMEOUT = timedelta(hours=15)


CHAT_NAME = "yabs_server_release_chat"
RECIPIENTS = {"telegram": {"chat_name": [CHAT_NAME]}, "yachats": {"chat_name": [CHAT_NAME]}}
NOTIFICATION_TEMPLATE_NAME = 'release_ticket_changed.j2'


class YabsServerNotifyReleaseTicketChanged(BaseTask):
    name = 'YABS_SERVER_NOTIFY_RELEASE_TICKET_CHANGED'

    class Parameters(BaseTask.Parameters):
        ticket_filed = sdk2.parameters.String("ST Ticket Filed", default='status.key')
        ticket_filed_value = sdk2.parameters.String("ST Ticket Filed Value", default='readyToDeploy')

        release_confirm_people = sdk2.parameters.List("People, who can confirm releases (logins)")
        notification_is_enabled = sdk2.parameters.Bool("Notify about change", default=True)
        wait_until_change = sdk2.parameters.Bool("Wait when there is no change", default=False)

    class Context(sdk2.Context):
        termination_time = None

    def on_create(self):
        self.Context.termination_time = (datetime.now() + WAIT_TIMEOUT).strftime("%Y-%m-%dT%H:%M")

    def on_execute(self):
        release_ticket_key = self.get_release_ticket()
        ticket_filed_value = self.__get_st_issue_filed_value(release_ticket_key)

        if self.Parameters.ticket_filed_value == ticket_filed_value:
            self.__notify(release_ticket_key)
        else:
            self.__wait()

    def __wait(self):
        if self.Parameters.wait_until_change and (datetime.now()).strftime("%Y-%m-%dT%H:%M") < self.Context.termination_time:
            raise sdk2.WaitTime(WAIT_TIME)

    def __get_st_issue_filed_value(self, release_ticket_key):
        st_client = self.__get_st_client(self.__get_token_from_vault(self.Parameters.st_vault_name))
        issue = st_client.issues[release_ticket_key]

        ticket_filed_value = self.__get_attribute_value(issue, self.Parameters.ticket_filed)
        logging.info('Current {} = {}'.format(self.Parameters.ticket_filed, ticket_filed_value))

        return ticket_filed_value

    def __get_attribute_value(self, issue, path_to_attribute):
        attributes = path_to_attribute.split('.')
        value = issue
        for attribute in attributes:
            value = getattr(value, attribute)

        return value

    def __get_st_client(self, st_token):
        from startrek_client import Startrek
        return Startrek(useragent='YABS_SERVER_NOTIFY_RELEASE_TICKET_CHANGED', base_url='https://st-api.yandex-team.ru', token=st_token)

    def __get_on_duty_users(self):
        on_duty_users = []
        try:
            on_duty_users = list(set([self.author, AbcClient(self.__get_abc_token()).get_current_duty_login(179, schedule_slug='yabs_frontend_duty_first')]))
        except Exception:
            self.set_info('Cannot get On Duty Users')
            logging.error('Cannot get On Duty Users', exc_info=True)

        return on_duty_users

    def __get_notification_template(self):
        template_dir = os.path.join(os.path.dirname(__file__), '..', '..', 'notifications', 'templates')
        try:
            # Non binary task is executed here "/home/zomb-sandbox/tasks/projects/yabs/release/tasks/YabsServerGetB2BDiffs/__init__.py"
            return get_template(NOTIFICATION_TEMPLATE_NAME, templates_dir=template_dir)
        except Exception:
            logging.error('Cannot find Notification Template in {}'.format(template_dir), exc_info=True)

            # In case of local build
            template_dir = 'sandbox/projects/yabs/release/notifications/templates/'
            logging.info('Try to load template in {}'.format(template_dir))
            return get_template(NOTIFICATION_TEMPLATE_NAME, templates_dir=template_dir)

    def __notify(self, release_ticket_key):
        if not self.Parameters.notification_is_enabled:
            logging.info("Notification is disabled in Sandbox Task parameters")
            return

        release_confirm_users = self.Parameters.release_confirm_people
        release_confirm_users.extend(self.__get_on_duty_users())

        notification_template = self.__get_notification_template()

        for transport, recipient in RECIPIENTS.items():
            mentions = get_logins(self, release_confirm_users, transport=transport)

            try:
                message = notification_template.render({'release_ticket': release_ticket_key,
                                                        'release_ticket_link' : "http://st.yandex-team.ru/{}".format(release_ticket_key),
                                                        'transport': transport,
                                                        'ticket_filed': self.Parameters.ticket_filed,
                                                        'ticket_filed_value': self.Parameters.ticket_filed_value,
                                                        'on_duty_users': mentions})

                logging.debug(message)
                send_message(message, self.__get_juggler_token(), recipients={transport: recipient})
            except Exception:
                self.set_info('Cannot send notification')
                logging.error('Cannot send notification', exc_info=True)

    def __get_token(self, token_name):
        tokens = sdk2.yav.Secret(YAV_SECRET).data()
        return tokens[token_name]

    def __get_juggler_token(self):
        return self.__get_token('juggler_token')

    def __get_abc_token(self):
        return self.__get_token('abc_token')

    def __get_token_from_vault(self, vault_name):
        return sdk2.Vault.data(vault_name)
