#!/usr/bin/env python
# -*- coding: utf8 -*-

import logging
import os

from sandbox import sdk2
from sandbox.projects.yabs.release.common import BaseReleaseTask
from sandbox.projects.yabs.release.notifications.jns.client import send_message
from sandbox.projects.yabs.release.notifications.jns.helpers import get_logins
from sandbox.projects.yabs.release.notifications.environment.report_info import BaseReportData, IssueInfo
from sandbox.projects.yabs.qa.template_utils import get_template


logger = logging.getLogger(__name__)

TEMPLATES_DIR = os.path.join(os.path.dirname(__file__), '..', '..', '..', 'release', 'notifications', 'templates')
TEMPLATE_NAME = 'ready_for_stable.j2'


class ReportData(BaseReportData):
    __slots__ = BaseReportData.__slots__ + ('issue', )

    def __init__(
            self,
            issue=None,
            **kwargs
    ):
        super(ReportData, self).__init__(**kwargs)

        self.tags = self.tags + ['ready']
        self.issue = IssueInfo(issue)


class YabsServerReadyForStableNotify(BaseReleaseTask):

    class Requirements(BaseReleaseTask.Requirements):
        cores = 1
        ram = 1024

        class Caches(BaseReleaseTask.Requirements.Caches):
            pass

    class Parameters(sdk2.Parameters):
        release_confirm_people = sdk2.parameters.List("People, who can confirm releases (logins)", default=['poldnev', 'fdrstrok'])
        component_name = BaseReleaseTask.Parameters.component_name()
        release_number = BaseReleaseTask.Parameters.release_number()
        with sdk2.parameters.Group("Startrek") as startrek_params:
            release_ticket = BaseReleaseTask.Parameters.release_ticket()
            st_vault_name = BaseReleaseTask.Parameters.st_vault_name()

    def get_recipients(self):
        if self.Context.copy_of:
            return {
                "yachats": {"internal": [{"login": self.author}]},
                "telegram": {"internal": [{"login": self.author}]},
            }
        return {
            "telegram": {"chat_name": ["yabs_server_release_chat"]},
            "yachats": {"chat_name": ["yabs_server_release_chat"]},
        }

    def _notify_event(self, report_data):
        report_template_j2 = get_template(TEMPLATE_NAME, templates_dir=TEMPLATES_DIR)

        tokens = sdk2.yav.Secret("sec-01fx7jcsjevejnypw63tk26nj3").data()
        try:
            spawn_users = self.Parameters.release_confirm_people
            for transport, recipient in self.get_recipients().items():
                mentions = get_logins(self, spawn_users, transport=transport)
                report = report_template_j2.render(report_data.as_dict(transport=transport, mentions=mentions))
                send_message(report, tokens['juggler_token'], recipients={transport: recipient})
        except Exception:
            self.set_info('Cannot send notification')
            logger.error('Cannot send notification', exc_info=True)

    def on_execute(self):
        report_data = ReportData(
            task_id=self.id,
            task_type=str(self.type),
            component_name=self.Parameters.component_name,
            major_version=self.Parameters.release_number,
            issue=self.get_release_ticket(),
        )
        self._notify_event(report_data)
