import logging
import traceback
from collections import defaultdict, namedtuple

from concurrent.futures import as_completed, ThreadPoolExecutor

from sandbox.projects.common.nanny.client import NannyClient


logger = logging.getLogger(__name__)
NannyResourceInfo = namedtuple("NannyResourceInfo", ("deploy_time", "build_task_id", "resource_id"))

NANNY_BASE_URL = 'http://nanny.yandex-team.ru/'


class NannyHelper(object):
    def __init__(self, nanny_client=None, nanny_token=None, nanny_base_url=NANNY_BASE_URL, *args, **kwargs):
        if nanny_client is None:
            nanny_client = NannyClient(nanny_base_url, nanny_token)
        self.__nanny_client = nanny_client
        self.__service_labels_cache = {}

    def get_service_labels(self, service_id):
        labels = self.__service_labels_cache.get(service_id)
        if labels is not None:
            return labels
        service_info = self.__nanny_client.get_service(service_id)
        # Note: label keys not unique, so last value will be used
        labels = dict([(label['key'], label.get('value')) for label in service_info['info_attrs']['content']['labels']])
        self.__service_labels_cache[service_id] = labels
        return labels

    def get_service_label(self, service_id, label_name):
        return self.get_service_labels(service_id).get(label_name)

    def get_last_runtime_attrs(self, service_id):
        target_state = self.__nanny_client.get_service_target_state(service_id)
        snapshot_id = target_state['content']['snapshot_id']
        runtime_attrs = self.__nanny_client.get_history_runtime_attrs(snapshot_id)
        return runtime_attrs

    def get_services_active_runtime_attrs(self, dashboard, labels=None):
        info = defaultdict(list)
        service_ids = self.__nanny_client.get_dashboard_services(dashboard)
        results = {}
        with ThreadPoolExecutor(max_workers=16) as executor:
            for service_id in service_ids:
                if labels:
                    skip_service = False
                    for label_name, label_value in labels.items():
                        if self.get_service_label(service_id, label_name) != label_value:
                            skip_service = True
                    if skip_service:
                        continue

                current_state = self.__nanny_client.get_service_current_state(service_id)
                logger.debug('Got %s state: %s', service_id, current_state)
                if current_state['content']['summary']['value'] == 'ONLINE':
                    results[executor.submit(self.get_last_runtime_attrs, service_id)] = service_id

        for future in as_completed(results):
            service_id = results[future]
            exc = future.exception()
            if exc is not None:
                logger.error(exc)
                info[service_id] = None
            else:
                active_runtime_data = future.result()
                info[service_id] = active_runtime_data

        return info

    @classmethod
    def get_resource_info(cls, active_runtime_data, resource_type):
        deploy_time = active_runtime_data.get("change_info", {}).get("ctime", 0) / 1000
        resources_info = [
            x for x in active_runtime_data['content']['resources']['sandbox_files']
            if x['resource_type'] == resource_type
        ]
        try:
            first_resource = resources_info[0]
        except IndexError:
            logger.error('No resource %s found: %s', resource_type, active_runtime_data['content']['resources']['sandbox_files'])
            raise

        task_id = first_resource['task_id']
        resource_id = first_resource['resource_id']
        return NannyResourceInfo(deploy_time, task_id, resource_id)

    @classmethod
    def get_resource_info_by_service(cls, services_active_runtime_attrs, resource_type):
        resources_info = {}
        for service_id, active_runtime_data in services_active_runtime_attrs.items():
            try:
                resource_info = cls.get_resource_info(active_runtime_data, resource_type)
            except Exception:
                logger.error("Cannot get resource info for '%s': %s", service_id, traceback.format_exc())
                resource_info = NannyResourceInfo(0, None, None)

            resources_info[service_id] = resource_info

        return resources_info

    def get_services_resource_info(self, dashboard, resource_type, labels=None):
        services_active_runtime_attrs = self.get_services_active_runtime_attrs(dashboard, labels=labels)
        resources_info = {}
        for service_id, active_runtime_data in services_active_runtime_attrs.items():
            try:
                resource_info = self.get_resource_info(active_runtime_data, resource_type)
            except Exception as exc:
                logger.error("Cannot get resource info for '%s': %s", service_id, exc)
                resource_info = NannyResourceInfo(0, None, None)

            resources_info[service_id] = resource_info

        return resources_info
