import logging
import re

from sandbox.common.rest import Client
from sandbox.common.proxy import NoAuth

from .version import ServerVersion, BasicVersion

logger = logging.getLogger(__name__)


TAG_FIELD_NAME = "checkout_arcadia_from_url"
TAG_FIELD_FORMAT = r"[^\d]*(?P<major>\d+)-(?P<minor>\d+).*"

FULL_VERSION_FIELD_NAME = "version"
FULL_VERSION_FIELD_FORMAT = r"(?P<revision>\d+).(?P<basever>\d+).*"


def _extract_version_from_string(str_value, str_format):
    """
    >>> _extract_version_from_string("arcadia:/arc/tags/yabs_server_validator/stable-11-1/arcadia", TAG_FIELD_FORMAT)
    {'major': '11', 'minor': '1'}
    >>> _extract_version_from_string("arcadia:/arc/tags/yabs_server_l7balancer/stable-7-1/arcadia", TAG_FIELD_FORMAT)
    {'major': '7', 'minor': '1'}
    >>> _extract_version_from_string("arcadia:/arc/tags/yabs_server/r353-3/arcadia", TAG_FIELD_FORMAT)
    {'major': '353', 'minor': '3'}
    >>> _extract_version_from_string("arcadia:/arc/tags/yabs_server/r353-34/arcadia", TAG_FIELD_FORMAT)
    {'major': '353', 'minor': '34'}
    """
    m = re.search(str_format, str_value)
    try:
        version_dict = m.groupdict()
        logger.debug("Parsed version from tag: %s", version_dict)
    except AttributeError:
        raise ValueError("Cannot extract version from '{}', expected '{}'".format(str_value, str_format))
    return version_dict


class SandboxHelper(object):
    def __init__(self, sandbox_client=None, *args, **kwargs):
        if sandbox_client is None:
            sandbox_client = Client(base_url=Client.DEFAULT_BASE_URL, auth=NoAuth())
        self.__sandbox_client = sandbox_client

    def _parse_field_from_task(self, task_id, field_name, field_format):
        try:
            task_context = self.__sandbox_client.task[task_id].context.read()
            field_str = task_context[field_name]
        except KeyError:
            task_parameters = self.__sandbox_client.task[task_id].read()["input_parameters"]
            field_str = task_parameters[field_name]

        logger.debug("Found %s: %s", field_name, field_str)

        parsed_field = _extract_version_from_string(field_str, field_format)
        return parsed_field

    def get_basic_version_from_task(
        self,
        build_task_id,
        tag_field_name=TAG_FIELD_NAME,
        tag_field_format=TAG_FIELD_FORMAT,
    ):
        try:
            parsed_tag = self._parse_field_from_task(build_task_id, tag_field_name, tag_field_format)
            major, minor = int(parsed_tag["major"]), int(parsed_tag["minor"])
        except Exception as e:
            logger.exception(e)
            major, minor = 0, 0

        return BasicVersion(major, minor)

    def get_full_version_from_task(
        self,
        build_task_id,
        tag_field_name=TAG_FIELD_NAME,
        tag_field_format=TAG_FIELD_FORMAT,
        full_version_field_name=FULL_VERSION_FIELD_NAME,
        full_version_field_format=FULL_VERSION_FIELD_FORMAT,
    ):
        try:
            parsed_tag = self._parse_field_from_task(build_task_id, tag_field_name, tag_field_format)
            major, minor = int(parsed_tag["major"]), int(parsed_tag["minor"])
        except Exception as e:
            logger.exception(e)
            major, minor = 0, 0
        try:
            parsed_full_version = self._parse_field_from_task(build_task_id, full_version_field_name, full_version_field_format)
            base_version = int(parsed_full_version["basever"])
        except Exception as e:
            logger.exception(e)
            base_version = 0

        return ServerVersion(major, minor, base_version)

    def get_full_version_from_resource(self, resource_id):
        resource = self.__sandbox_client.resource[resource_id].read()
        return self.get_full_version_from_task(resource["task"]["id"])

    def get_full_version_from_resources(self, resource_ids):
        resource_ids = set(resource_ids)
        resources = self.__sandbox_client.resource.read(id=[_f for _f in resource_ids if _f], limit=len(resource_ids))
        versions = {
            resource["id"]: self.get_full_version_from_task(resource["task"]["id"])
            for resource in resources["items"]
        }
        if None in resource_ids:
            versions[None] = ServerVersion(0, 0, 0)
        return versions

    def get_resource_state(self, resource_id):
        resource = self.__sandbox_client.resource[resource_id].read()
        return resource["attributes"].get("released", "unknown")
