import logging
import six

from .version import ServerVersion, ServiceStage
from .sandbox_helpers import SandboxHelper
from .nanny_helpers import NannyHelper

logger = logging.getLogger(__name__)


class ServerVersionHelper(SandboxHelper, NannyHelper):
    def __init__(self, *args, **kwargs):
        SandboxHelper.__init__(self, *args, **kwargs)
        NannyHelper.__init__(self, *args, **kwargs)

    def get_stage_label(self, service_id):
        stage_label = self.get_service_label(service_id, "stage")
        if stage_label is None:
            # TODO: remove and use nanny labels
            for stage in (ServiceStage.mirror, ServiceStage.asan, ServiceStage.prestable, ServiceStage.stable, ServiceStage.experiment, ServiceStage.production):
                logger.debug("Trying to find stage %s for service %s", stage, service_id)
                if six.text_type(stage) in six.text_type(service_id):
                    logger.debug("Service %s is %s", service_id, stage)
                    return stage
            return ServiceStage.unknown
        return stage_label

    def get_deployed_version(self, service_id, resource_type):
        service_runtime_attrs = self.get_last_runtime_attrs(service_id)
        resource_info = self.get_resource_info(service_runtime_attrs, resource_type)
        try:
            server_version = self.get_full_version_from_task(resource_info.build_task_id)
            return server_version
        except Exception:
            logger.exception("Cannot get version info for '%s'", service_id)
            return ServerVersion(0, 0, 0)

    def get_deployed_versions(self, dashboard, resource_type, labels=None):
        services_resource_info = self.get_services_resource_info(dashboard, resource_type, labels=labels)
        versions = {}
        build_tasks_by_service = {}
        for service_id, resource_info in services_resource_info.items():
            build_tasks_by_service[service_id] = resource_info.build_task_id

        versions_by_build_task = {}
        for build_task_id in set(build_tasks_by_service.values()):
            try:
                server_version = self.get_full_version_from_task(build_task_id)
            except Exception:
                logger.exception("Cannot get version info for '%s'", service_id)
                server_version = ServerVersion(0, 0, 0)

            versions_by_build_task[build_task_id] = server_version

        for service_id, resource_info in services_resource_info.items():
            versions[service_id] = versions_by_build_task[resource_info.build_task_id]

        return versions

    def get_deployed_versions_info(self, dashboard, resource_type, labels=None):
        versions_by_service = self.get_deployed_versions(dashboard, resource_type, labels=labels)
        services_resource_info = self.get_services_resource_info(dashboard, resource_type, labels=labels)
        versions = {}
        for service_id, server_version in versions_by_service.items():
            resource_info = services_resource_info[service_id]
            if server_version not in versions:
                versions[server_version] = {
                    "deploy_time": resource_info.deploy_time,
                    "nanny_services": [],
                    "build_task_id": resource_info.build_task_id,
                    "resource_id": resource_info.resource_id,
                    "labels": self.get_service_labels(service_id),
                }

            versions[server_version]["nanny_services"].append(service_id)
            versions[server_version]["deploy_time"] = max(resource_info.deploy_time, versions[server_version]["deploy_time"])

        return versions
