import logging
from sandbox import sdk2

import sandbox.common.types.task as ctt
import sandbox.projects.common.build.parameters as cbp
import sandbox.projects.common.build.YaPackage as YaPackage
from sandbox.common.types.misc import NotExists
import datetime


class AuxService():
    def __init__(self, name, nanny_path, package_path):
        self.name = name
        self.nanny_path = nanny_path
        self.package_path = package_path


service_dict = {
    'poormansprofiler': AuxService('poor man`s profiler', 'poormansprofiler.tgz', 'yabs/poormansprofiler/yabs-poormansprofiler.json'),
    'validator': AuxService('yabs-server-base-validator', 'yabs-server-base-validator.tar.gz', 'yabs/server/infra/validator/yabs-server-base-validator.json'),
    'trivial_cron': AuxService('yabs trivial cron', 'yabs-trivial-cron.tar.gz', 'yabs/server/infra/trivial_cron/package.json'),
    'validator_hooks': AuxService('yabs-server-base-validator-hooks', 'yabs-server-base-validator-hooks.tar.gz', 'yabs/server/infra/validator/yabs-server-base-validator-hooks.json'),
    'bstr': AuxService('bstr (yabs-transport-ng)', 'yabs-transport-ng.tar.gz', 'yabs/server/infra/bstr/yabs-transport-ng.json'),
    'rtc_logs': AuxService('server-rtc-logs', 'server-runtime-cloud-logs.tar.gz', 'yabs/server/infra/logs/package.json'),
    'event_utils': AuxService('yabs-event-utils', 'yabs-event-utils.tar.gz', 'yabs/event-utils/yabs-event-utils.json'),
    'logbroker_stat_sender': AuxService('yabs-logbroker-stat-sender', 'yabs-logbroker-stat-sender.tar.gz', 'yabs/utils/yabs-logbroker-stat-sender/yabs-logbroker-stat-sender.json'),
    'rt_graphs': AuxService('rt-graphs Perl package', 'rt-graphs.tar.gz', 'yabs/server/infra/rt_graphs/cloud/package.json'),
    'rtgraph_handle': AuxService('rtgraph parser (rtgraph-handle)', 'rtgraph-handle.tar.gz', 'yabs/server/infra/rtgraph/yabs-rtgraph-handle-tarball.json'),
    'coredump_watcher': AuxService('yabs-server-rtc-coredump-watch', 'server-rtc-coredump-watch.tgz', 'yabs/server/infra/coredump_watch/package.json'),
    'runtime_mon': AuxService('yabs-runtime-mon', 'yabs-runtime-mon.tar.gz', 'yabs/utils/yabs-runtime-mon/package-ng.json')
}


class YabsAuxServiceDeploy(sdk2.Task):
    description = ''

    execution_space = 100
    required_ram = 100

    class Parameters(sdk2.Task.Parameters):
        with sdk2.parameters.String('Aux service name', required=True) as AuxServiceName:
            for key in service_dict:
                AuxServiceName.values[key] = service_dict[key].name
        NannyPath = sdk2.parameters.String(
            'Nanny Path',
            description='Local path of resource in Nanny service container e.g. rtgraph-handle.tar.gz (defined automatically by aux service name if not set)',
            default=None,
        )
        ArcadiaPath = sdk2.parameters.String(
            'Arcadia Path',
            description='Path to package to be build e.g. yabs/server/infra/rtgraph/yabs-rtgraph-handle.json (defined automatically by aux service name if not set)',
            default=None,
        )
        ArcadiaRevision = sdk2.parameters.Integer('Arcadia Revision', description='Revision. trunk by default', default=None)
        NannyTokenVaultName = sdk2.parameters.String('Vault name of Nanny token', description='Nanny token vault name to manage Nanny services', default='robot-bs-autoadmin-nanny-token')
        StartrekTokenVaultName = sdk2.parameters.String('Vault name of Startrek token', description='Startrek token vault name to send comment', default='robot-bs-autoadmin-startrek-token')

    class Context(sdk2.Task.Context):
        _all_subtasks = []

    def run_ya_package(self):
        if self.Context.subtask_ya_package is NotExists:
            arcadia_url = cbp.ArcadiaUrl.default_value
            if self.Parameters.ArcadiaRevision is not None and self.Parameters.ArcadiaRevision > 0:
                arcadia_url += '@{}'.format(self.Parameters.ArcadiaRevision)
            subtask = sdk2.Task['YA_PACKAGE']
            subtask = subtask(
                self,
                description='Build resource for service {}'.format(service_dict[self.Parameters.AuxServiceName].name),
                **{
                    cbp.ArcadiaUrl.name: arcadia_url,
                    YaPackage.PackagesParameter.name: service_dict[self.Parameters.AuxServiceName].package_path
                    if self.Parameters.ArcadiaPath is None or self.Parameters.ArcadiaPath == ''
                    else self.Parameters.ArcadiaPath,
                    YaPackage.PackageTypeParameter.name: YaPackage.TARBALL,
                    cbp.BuildType.name: YaPackage.RELEASE,
                    YaPackage.UseNewFormatParameter.name: True,
                    cbp.DoNotRemoveResources.name: True
                }
            ).enqueue()
            self.Context.subtask_ya_package = subtask.id
            self.wait_subtasks([subtask.id])
        else:
            pass

    def wait_subtasks(self, subtask_list):
        self.Context._all_subtasks.extend(subtask_list)

        raise sdk2.WaitTask(
            tasks=self.Context._all_subtasks,
            statuses=[ctt.Status.Group.FINISH, ctt.Status.Group.BREAK],
            wait_all=True
        )

        if any(sdk2.Task[x].status not in ctt.Status.Group.SUCCEED for x in self.Context._all_subtasks):
            raise Exception("One of subtasks failed!")

    def create_startrek_ticket(self, title, info):
        logging.info('Try to create Startrek ticket')
        from projects.common.geosearch.startrek import StartrekClient
        token = sdk2.Vault.data(self.Parameters.StartrekTokenVaultName)
        self.startrek_client = StartrekClient(token)
        issue_key = self.startrek_client.create_task({
            'queue': 'BSRELEASE',
            'assignee': self.author,
            'followers': ['rmcf'],
            'summary': title,
            'description': info
        })
        logging.info('Created Startrek issue: {}'.format(issue_key))
        return issue_key

    def send_startrek_report(self, st_ticket, st_report):
        logging.info('Try to add Startrek comment')
        from projects.common.geosearch.startrek import StartrekClient
        token = sdk2.Vault.data(self.Parameters.StartrekTokenVaultName)
        self.startrek_client = StartrekClient(token)
        self.startrek_client.add_comment(st_ticket, st_report)

    def make_bump_draft(self):
        if self.Context.bump_task is NotExists:
            subtask = sdk2.Task['YABS_BUMP_AUX_SERVICE']
            subtask = subtask(
                self,
                description='Bump service {} in Nanny'.format(service_dict[self.Parameters.AuxServiceName].name),
                Path=service_dict[self.Parameters.AuxServiceName].nanny_path if self.Parameters.NannyPath is None or self.Parameters.NannyPath == '' else self.Parameters.NannyPath,
                Resource=sdk2.Resource['YA_PACKAGE'].find(
                    task=sdk2.Task[self.Context.subtask_ya_package]
                ).first(),
                NannyTokenVaultName=self.Parameters.NannyTokenVaultName,
                StartrekTokenVaultName=self.Parameters.StartrekTokenVaultName,
                STTicket=self.Context.st_ticket
            ).save()
            self.Context.bump_task = subtask.id

    def on_execute(self):
        # client = rest.Client()
        # self.check_children_statuses(client)

        logging.info("Start")

        self.run_ya_package()
        package_res = sdk2.Resource['YA_PACKAGE'].find(task=sdk2.Task[self.Context.subtask_ya_package]).first()
        if self.Context.st_ticket is NotExists:
            self.Context.st_ticket = self.create_startrek_ticket(
                'Release of {} rev. {} by {}'.format(service_dict[self.Parameters.AuxServiceName].name, package_res.svn_revision, datetime.datetime.now().strftime('%d.%m.%Y')),
                'YA_PACKAGE Sandbox task ID: {task} (https://sandbox.yandex-team.ru/task/{task}/view)\nResource ID: {resource} (https://sandbox.yandex-team.ru/resource/{resource}/view)'.format(
                    task=self.Context.subtask_ya_package, resource=package_res.id))
        self.make_bump_draft()
        report = 'https://sandbox.yandex-team.ru/task/{}/view'.format(self.Context.bump_task)
        self.send_startrek_report(self.Context.st_ticket, report)
        self.set_info(
            'Generated Startrek ticket: <a href="https://st.yandex-team.ru/{key}">{key}</a>\nYABS_BUMP_AUX_SERVICE task: <a href="https://sandbox.yandex-team.ru/task/{task}/view">{task}</a>'.format(
                key=self.Context.st_ticket,
                task=self.Context.bump_task
            ), do_escape=False,
        )
