# -*- coding: utf-8 -*-

import logging
import textwrap
import os
import re

from shutil import copyfile
from sandbox import sdk2

import boto3

import sandbox.sandboxsdk.process as process
import sandbox.projects.common.arcadia.sdk as arcadia_sdk

S3_ENDPOINT="http://s3.mdst.yandex.net"
S3_BUCKET="bin"
TMP_PATH="/tmp/bin"
TTL_DAYS=10
YAV_PATTERN = r'\$\(yav:(?P<sec>sec-.+):(?P<key>.+)\)'


class RunBinCached(sdk2.Task):
    """
    Builds and runs binary from SVN for simple periodic tasks.
    When revision is set in SVN path, the binary is cached on (svn_path,svn_revision,binary_path).
    Can not be used for builds that have side effects since only binary is cached.
    Does not fail if cache is unavailable
    """

    class Requirements(sdk2.Requirements):
        cores = 1
        ram = 4096
        disk_space = 8 * 1024

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Parameters):
        cache_secret_desc = textwrap.dedent('''Cache secret. If you start this task under your credentials, provide YAV secret containing fields: s3_id & s3_token (used for caching binary blob)''')
        cache_secret = sdk2.parameters.YavSecret(cache_secret_desc, default='sec-01es6fzte74n1wbd7z453wgr4d')
        svn_url = sdk2.parameters.String("Svn url", required=True)
        bin_path = sdk2.parameters.String("Binary to build and run", required=True)
        bin_args = sdk2.parameters.String("Arguments", required=False)
        bin_env_desc = textwrap.dedent('''Space separated environment variables: VAR1=val1 VAR2=val2. Can be used with yav secrets $(yav:sec-id:key): YT_TOKEN=$(yav:sec-cafebebe:yt_token)''')
        bin_env = sdk2.parameters.String(bin_env_desc, required=False)

    def on_execute(self):
        logging.info("Start")

        secret1 = self.Parameters.cache_secret.data()
        s3_id = secret1["s3_id"]
        s3_token = secret1["s3_token"]

        secret2 = sdk2.yav.Secret("sec-01eh75y361z96f72h1bbcy3m5b").data()
        arc_token = secret2["arc_token"]

        client = boto3.client(
            's3',
            endpoint_url=S3_ENDPOINT,
            aws_access_key_id=s3_id,
            aws_secret_access_key=s3_token,
        )

        logging.info("'svn url': %s", self.Parameters.svn_url)

        try:
            ttl = {
                'Rules': [{
                    'ID': 'My rule',
                    'Expiration': {
                        'Days': TTL_DAYS,
                    },
                    'Filter': {'Prefix': '/'},
                    'Status': 'Enabled'
                }
                ]}
            client.put_bucket_lifecycle_configuration(Bucket=S3_BUCKET, LifecycleConfiguration=ttl)
        except Exception as e:
            logging.error("applying ttl: %s", e)

        if not self.try_cached(client, arc_token):
            with arcadia_sdk.mount_arc_path(self.Parameters.svn_url, use_arc_instead_of_aapi=True, arc_oauth_token=arc_token) as arcadia:
                logging.info("arcadia mounted: %s", arcadia)
                bin_path = os.path.join(arcadia, self.Parameters.bin_path)
                logging.info("bin path: %s", bin_path)

                copyfile(TMP_PATH, bin_path)
                self.exec_bin(bin_path, arcadia)

        logging.info("all done")

    def try_cached(self, client, arc_token):
        with arcadia_sdk.mount_arc_path(self.Parameters.svn_url, use_arc_instead_of_aapi=True, arc_oauth_token=arc_token) as arcadia:
            logging.info("arcadia mounted: %s", arcadia)
            bin_path = os.path.join(arcadia, self.Parameters.bin_path)
            logging.info("bin path: %s", bin_path)

            s3_key = self.Parameters.svn_url + "/" + self.Parameters.bin_path
            logging.info("S3 key: %s", s3_key)

            done = False

            if not ("@" in self.Parameters.svn_url):
                logging.info("using HEAD, caching disabled")
                self.build(arcadia)
                copyfile(bin_path, TMP_PATH)
            else:
                try:
                    client.download_file(S3_BUCKET, s3_key, bin_path)
                    logging.info("binary downloaded ok")
                    self.exec_bin(bin_path, arcadia)
                    done = True
                except Exception as e:
                    logging.warning("download binary: '%s' building...", e)
                    self.build(arcadia)
                    copyfile(bin_path, TMP_PATH)
                    try:
                        client.upload_file(bin_path, S3_BUCKET, s3_key)
                        logging.info("binary uploaded ok")
                    except Exception as e:
                        logging.error("upload binary: '%s'", e)
            return done

    def deref_yav(self, s):
        def get_yav(match):
            secret = sdk2.yav.Secret(match.group('sec')).data()
            return secret[match.group('key')]

        return re.sub(YAV_PATTERN, get_yav, s)

    def build(self, arcadia):
        build_target = os.path.dirname(self.Parameters.bin_path)
        logging.info("build target: %s", build_target)
        build_return_code = arcadia_sdk.do_build('ya', arcadia, [build_target], clear_build=False, results_dir=arcadia)
        logging.info("build errno: %s", build_return_code)

    def exec_bin(self, bin_path, work_dir):
        logging.info("running %s in %s", bin_path, work_dir)
        env = {}
        os.chmod(bin_path, 0o777)
        if self.Parameters.bin_env:
            vars = [self.deref_yav(k) for k in self.Parameters.bin_env.split()]
            env = {elem[0]: elem[1] for elem in (var.split("=") for var in vars)}
            logging.info("env: %s", env)
        status = process.run_process(
            [bin_path] + self.Parameters.bin_args.split(),
            log_prefix='program',
            outputs_to_one_file=False,
            shell=True,
            work_dir=work_dir,
            environment=env,
            )
        logging.info("binary errno: %d", status.returncode)
