'''
Sandbox task tracing library.
'''

from __future__ import absolute_import, division, print_function

import logging
import os

from sandbox.projects.yabs.sandbox_task_tracing.impl.context_managers import (
    flush_trace as __flush_trace,
    EntryPoint,
    TraceContext,
    TraceNewResources,
    TraceNewTasks,
)
from sandbox.projects.yabs.sandbox_task_tracing.info import (
    arguments_info,
    command_info,
    make_arguments_info_spec,
    set_tracing_files_prefix,
    symbol_info,
)
from sandbox.projects.yabs.sandbox_task_tracing.util import decorator_function, frozendict


logger = logging.getLogger(__name__)

set_tracing_files_prefix(os.path.dirname(__file__) + '/')


def flush_trace(timeout=None, ignore_errors=None):
    '''
    Flushes trace buffer.

    It is necessary, for instance, before using `multiprocessing.Pool`.

    :param float timeout: flush timeout in seconds, does not include lock wait time
        (default is `DEFAULTS['entry_point_spec']['flush_timeout_seconds']`)
    :param bool ignore_errors: if true, errors (including timeout) will be ignored
        (default is `DEFAULTS['entry_point_spec']['ignore_exit_errors']`)
    :raises: exceptions.BufferTimeout: if flush could not complete within the specified time
    '''
    return __flush_trace(timeout=timeout, ignore_errors=ignore_errors)


def trace(name, info=frozendict(), stack_info_enabled=None, stack_info_spec=frozendict()):
    '''
    Returns context manager for tracing a block of code.

    Produces records of type `trace`.

    Resulting `info` field after conversion to JSON:
    ```json
    {
        "exception": <exception info>,                      # only if exception was raised, see `info.exception.exception_info`
        "name": <name>,                                     # value of the `name` argument
        "stack": <stack info>,                              # only if `stack_info_enabled`, see `info.stack.stack_info`
        "<key>": <value>,                                   # key and value from the `info` dict,
        ...                                                 # repeated for each dict item
    }
    ```
    Note that values eventually pass through 'jsonified'

    :param str name: unique name of the block
    :param dict info: arbitrary key-value pairs to be included in the result
    :param bool stack_info_enabled: if true, stack info will be added to the result
    :param dict stack_info_spec: stack info spec for `info.stack.stack_info`
    :return: context manager
    '''
    return TraceContext(
        'trace',
        info=dict(info, name=name),
        stack_info_enabled=stack_info_enabled,
        stack_info_spec=stack_info_spec,
    )


@decorator_function
def trace_calls(
    info=frozendict(),
    save_arguments=None,
    save_return_value=False,
    stack_info_enabled=None,
    stack_info_spec=frozendict(),
):
    '''
    Returns decorator for tracing calls to a function.

    Decorated function produces a record of type `call` each time it is called.

    Resulting `info` field after conversion to JSON:
    ```json
    {
        "arguments": <arguments info>                       # only if `save_arguments`, see `info.arguments.arguments_info`
        "exception": <exception info>,                      # only if exception was raised, see `info.exception.exception_info`
        "function": <function info>,                        # function symbol info, see `info.symbol.symbol_info`
        "return_value": <function return value>             # only if `save_return value`
        "stack": <stack info>,                              # only if `stack_info_enabled`, see `info.stack.stack_info`
        "<key>": <value>,                                   # key and value from the `info` dict,
        ...                                                 # repeated for each dict item
    }
    ```
    Note that values eventually pass through 'jsonified'

    :param dict info: arbitrary key-value pairs to be included in the result
    :param save_arguments: arguments to save, see `info.arguments.make_arguments_info_spec`
    :type save_arguments: Literal['all', 'positional', 'keyword'] | Container[int | str]
    :param bool save_return_value: if true, return value will be added to the result
    :param bool stack_info_enabled: if true, stack info will be added to the result
    :param dict stack_info_spec: stack info spec, see `info.stack.stack_info`
    :return: function decorator
    '''
    arguments_info_spec = make_arguments_info_spec(save_arguments) if save_arguments else None
    info = dict(info)  # make a shallow copy of info since we are going to change it

    def decorator(function):

        info.update(function=symbol_info(function))

        def wrapper(*args, **kwargs):

            with TraceContext(
                'call',
                info=info,
                stack_info_enabled=stack_info_enabled,
                stack_info_spec=stack_info_spec,
            ) as trace_context:

                if arguments_info_spec:
                    trace_context.info.update(arguments=arguments_info(args, kwargs, arguments_info_spec))

                return_value = function(*args, **kwargs)

                if save_return_value:
                    trace_context.info.update(return_value=return_value)

                return return_value

        return wrapper

    return decorator


@decorator_function
def trace_entry_point(
    writer_factory,
    spec=frozendict(),
    call_info=frozendict(),
    **call_kwargs
):
    '''
    Returns decorator for the `on_execute` method of the traced task.

    Decorated function when called produces a record of type `call` with additional entry `"entry_point": true`.

    Also, it initializes tracing writer and produces records for the task, current iteration and all preceding audit events.

    Resulting `info` field of the `tasks` table row is as returned by `info.task.task_info` with `format_='full'`.

    Resulting `info` field of the `iteration` record after conversion to JSON:
    ```json
    {
        "requirements": <current task requirements>         # as returned by Sandbox API
    }
    ```

    Resulting `info` field of the `audit` records after conversion to JSON:
    ```json
    {
        "record": <audit record>,                           # as returned by Sandbox API
        "semaphores": <requirements["semaphores"]>          # only if non-empty and `record["status"]` is `'ENQUEUED'`
    ```

    Resulting `info` field of the `call` record after conversion to JSON:
    ```json
    {
        "entry_point": true,
        ...                                                 # see `trace_calls` for the rest
    }
    ```

    :param Callable[[], AbstractTraceWriter] writer_factory: a callable returning new writer instance
    :param dict spec: entry point spec; supported keys:
        bool ignore_initialization_errors: if true, errors in tracing initialization will not fail the task;
        float flush_timeout_seconds: flush timeout in seconds;
        bool finish_on_exit: if true, exiting the context will finish writing thread (further trace records will be lost);
        float finish_timeout_seconds: finish timeout in seconds;
        bool ignore_exit_errors: if true, errors on exiting context will not fail the task;
    :param dict call_info: passed to `trace_calls` augmented with `"entry_point": true`
    :param dict call_kwargs: passed to `trace_calls`
    :return: function decorator
    '''
    def decorator(method):
        method = trace_calls(info=dict(call_info, entry_point=True), **call_kwargs)(method)
        def wrapper(task, *args, **kwargs):
            with EntryPoint(
                task=task,
                writer_factory=writer_factory,
                spec=spec,
            ) as trace_entry_point:
                with trace_entry_point.trace_current_audit_record:
                    return method(task, *args, **kwargs)
        return wrapper
    return decorator


def trace_new_resources(info=frozendict(), stack_info_enabled=None, stack_info_spec=frozendict()):
    '''
    Returns context manager for tracing resources creation.

    Produces records of type `new_resources`.

    Returned context manager provides the following additional list-like methods for registering new resources:
    * `append(self, resource)`: adds new resource to the new resources list
    * `extend(self, resources)`: adds new resources to the new resources list

    Resulting `info` field after conversion to JSON:
    ```json
    {
        "exception": <exception info>,                      # only if exception was raised, see `info.exception.exception_info`
        "resources": [
            <resource info>,                                # see `info.resource.resource_info`,
            ...                                             # repeated for each resource added using `append` or `extend`
        ],
        "stack": <stack info>,                              # only if `stack_info_enabled`, see `info.stack.stack_info`
        "<key>": <value>,                                   # key and value from the `info` dict,
        ...                                                 # repeated for each dict item
    }
    ```
    :param dict info: arbitrary key-value pairs to be included in the result
    :param bool stack_info_enabled: if true, stack info will be added to the result
    :param dict stack_info_spec: stack info spec for `info.stack.stack_info`
    :return: context manager
    '''
    return TraceNewResources(
        'new_resources',
        info=info,
        stack_info_enabled=stack_info_enabled,
        stack_info_spec=stack_info_spec,
    )


def trace_new_tasks(info=frozendict(), stack_info_enabled=None, stack_info_spec=frozendict()):
    '''
    Returns context manager for tracing tasks creation.

    Produces records of type `new_tasks`.

    Returned context manager provides the following additional list-like methods for registering new tasks:
    * `append(self, task)`: adds new task to the new tasks list
    * `extend(self, tasks)`: adds new tasks to the new tasks list

    Resulting `info` field after conversion to JSON:
    ```json
    {
        "exception": <exception info>,                      # only if exception was raised, see `info.exception.exception_info`
        "tasks": [
            <task info>,                                    # see `info.task.task_info`,
            ...                                             # repeated for each task added using `append` or `extend`
        ],
        "stack": <stack info>,                              # only if `stack_info_enabled`, see `info.stack.stack_info`
        "<key>": <value>,                                   # key and value from the `info` dict,
        ...                                                 # repeated for each dict item
    }
    ```
    :param dict info: arbitrary key-value pairs to be included in the result
    :param bool stack_info_enabled: if true, stack info will be added to the result
    :param dict stack_info_spec: stack info spec for `info.stack.stack_info`
    :return: context manager
    '''
    return TraceNewTasks(
        'new_tasks',
        info=info,
        stack_info_enabled=stack_info_enabled,
        stack_info_spec=stack_info_spec,
    )


def trace_subprocess(command, info=frozendict(), stack_info_enabled=None, stack_info_spec=frozendict()):
    '''
    Returns context manager for tracing external commands.

    Resulting `info` field after conversion to JSON:
    ```json
    {
        "command": <command>,                               # value of the `command` argument
        ...                                                 # see `trace` for the rest
    }
    ```
    :param list command: command being run
    :param dict info: arbitrary key-value pairs to be included in the result
    :param bool stack_info_enabled: if true, stack info will be added to the result
    :param dict stack_info_spec: stack info spec for `info.stack.stack_info`
    :return: context manager
    '''
    return TraceContext(
        'subprocess',
        info=dict(info, command=command_info(command)),
        stack_info_enabled=stack_info_enabled,
        stack_info_spec=stack_info_spec,
    )


@decorator_function
def trace_subprocess_calls(
    info=frozendict(),
    save_arguments=None,
    save_return_value=False,
    stack_info_enabled=None,
    stack_info_spec=frozendict(),
):
    '''
    Returns decorator for tracing calls to `subprocess.run`-like functions.

    Decorated function produces a record of type `subprocess` each time it is called.

    First argument of the wrapped function should contain the executed command.
    Iterator will be converted to list. Python 3 `bytes` objects are not supported.

    Resulting `info` field after conversion to JSON:
    ```json
    {
        "command": [...],                                   # info on the wrapped function first argument,
                                                            # as returned by `info.command.command_info`
        ...                                                 # see `trace_calls` for the rest
    }
    ```
    Note that values eventually pass through 'jsonified'

    :param dict info: arbitrary key-value pairs to be included in the result
    :param save_arguments: arguments to save, see `info.arguments.make_arguments_info_spec`
    :type save_arguments: Literal['all', 'positional', 'keyword'] | Container[int | str]
    :param bool save_return_value: if true, return value will be added to the result
    :param bool stack_info_enabled: if true, stack info will be added to the result
    :param dict stack_info_spec: stack info spec, see `info.stack.stack_info`
    :return: function decorator
    '''
    arguments_info_spec = make_arguments_info_spec(save_arguments) if save_arguments else None
    info = dict(info)  # make a shallow copy of info since we are going to change it

    def decorator(function):

        info.update(function=symbol_info(function))

        # first argument must be named `args` for `subprocess` module compatibility
        def wrapper(args, *other_args, **kwargs):

            with TraceContext(
                'subprocess',
                info=info,
                stack_info_enabled=stack_info_enabled,
                stack_info_spec=stack_info_spec,
            ) as trace_context:

                if not hasattr(args, '__len__'):  # we were passed an iterator?
                    args = list(args)

                trace_context.info.update(command=command_info(args))

                if arguments_info_spec:
                    trace_context.info.update(arguments=arguments_info((args,) + other_args, kwargs, arguments_info_spec))

                return_value = function(args, *other_args, **kwargs)

                if save_return_value:
                    trace_context.info.update(return_value=return_value)

                return return_value

        return wrapper

    return decorator
