'''
Information about function arguments.
'''

from __future__ import absolute_import, division, print_function

import collections
import six


__ArgumentsInfoSpec = collections.namedtuple('__ArgumentsInfoSpec', ('saved_args', 'saved_kwargs'))


def make_arguments_info_spec(save_arguments):
    '''
    Makes and returns arguments_info_spec for the `arguments_info`.

    :param save_arguments: either one of {'all', 'positional', 'keyword'}, or numbers and names of arguments to save
    :type save_arguments: Literal['all', 'positional', 'keyword'] | Container[int | str]
    :return: prepared save_arguments arguments_info_spec
    :rtype: object
    '''
    if isinstance(save_arguments, six.string_types):
        saved_args = 'all' if save_arguments in ('all', 'positional') else set()
        saved_kwargs = 'all' if save_arguments in ('all', 'keyword') else set()
        if not (saved_args or saved_kwargs):
            raise ValueError('Unsupported save_arguments: {!r}'.format(save_arguments))
    else:
        saved_args = set()
        saved_kwargs = set()
        for save_argument in save_arguments:
            if isinstance(save_argument, int):
                saved_args.add(save_argument)
            elif isinstance(save_argument, six.string_types):
                saved_kwargs.add(save_argument)
            else:
                raise ValueError('Unsupported save_arguments element type: {}'.format(type(save_argument).__name__))

    return __ArgumentsInfoSpec(saved_args=saved_args, saved_kwargs=saved_kwargs)


def arguments_info(args, kwargs, arguments_info_spec):
    '''
    Returns information about standard Python `(args, kwargs)` pair.

    Returned value after conversion to JSON:
    ```json
    {
        "args": {                                           # only if saving any positional argument is requested
            <value> or null,                                # unrequested values are represented by `null`s
            ...                                             # repeats for each element of `args`, requested or not
        },
        "kwargs": {                                         # only if saving any keyword argument is requested
            "<name>": <value>,
            ...                                             # repeats for each requested keyword argument present in `kwargs`
        }
    }
    ```
    Note that values eventually pass through 'jsonified'

    :param list args: `args` value
    :param dict kwargs: `kwargs` value
    :param object arguments_info_spec: value returned by `make_arguments_info_spec` call
    :return: tracing information in the format described above
    :rtype: dict
    '''

    result = {}

    saved_args = arguments_info_spec.saved_args
    if saved_args:
        result.update(args=[
            value if saved_args == 'all' or index in saved_args else None
            for index, value in enumerate(args)
        ])

    saved_kwargs = arguments_info_spec.saved_kwargs
    if saved_kwargs:
        result.update(kwargs={
            key: value
            for key, value in six.iteritems(kwargs)
            if saved_kwargs == 'all' or key in saved_kwargs
        })

    return result
