'''
Information about active exception.
'''

from __future__ import absolute_import, division, print_function

import traceback

try:
    from sandbox.sandboxsdk.task import MemoizeStage
except ImportError:
    class MemoizeStage(object):
        SkipStageException = Exception()

from sandbox.sdk2.helpers.misc import MemoizeStage as MemoizeStage2
from sandbox.sdk2.task import Wait

from sandbox.projects.yabs.sandbox_task_tracing.info.stack import stack_info
from sandbox.projects.yabs.sandbox_task_tracing.info.symbol import symbol_info


def exception_info(exception_type, exception_value, exception_traceback, stack_info_enabled, stack_info_spec):
    '''
    Returns information about standard Python `(exception_type, exception_value, exception_traceback)` trio.

    Returned value after conversion to JSON:
    ```json
    {
        "encoded_args": <exception_value.encode_args()>,    # only if exception is of type `sandbox.sdk2.task.Wait`
        "skip_stage": true,                                 # only if exception instance indicates skipped task stage
        "stack": <stack info>,                              # only if `stack_info_enabled`, see `stack.stack_info`
        "type": <exception type info>                       # as returned by `.symbol.symbol_info`
    }
    ```

    :param type exception_type: exception type
    :param BaseException exception_value: exception value
    :param list exception_traceback: exception traceback
    :param bool stack_info_enabled: if true, stack info will be added to the result
    :param dict stack_info_spec: stack info spec, see `.stack.stack_info`
    :return: tracing information in the format described above
    :rtype: dict
    '''

    result = dict(type=symbol_info(exception_type))

    if exception_value is MemoizeStage.SkipStageException or exception_value is MemoizeStage2.SkipStageException:
        result.update(skip_stage=True)

    if isinstance(exception_value, Wait):
        result.update(encoded_args=exception_value.encode_args())

    if stack_info_enabled:
        result.update(stack=stack_info(traceback.extract_tb(exception_traceback), stack_info_spec, is_exception=True))

    return result
