'''
Information about TestEnv/Sandbox GSID context attribute.
'''

from __future__ import absolute_import, division, print_function


class GsidWordParseError(Exception):
    '''
    Error parsing one of the GSID words.
    '''
    pass


def __gsid_element_info(fields, data):
    components = data.split(':')
    if len(fields) != len(components):
        raise GsidWordParseError('Invalid number of components: expected={}, found={}'.format(len(fields), len(components)))
    return dict(zip(fields, components))


def __gsid_int(value):
    try:
        return int(value)
    except ValueError as error:
        raise GsidWordParseError(*error.args)


def gsid_info(text, ignore_errors=True):
    '''
    Returns information about TestEnv/Sandbox GSID context attribute.

    Returned value after conversion to JSON:
    ```json
    {
        "_errors": [                                        # only if there are parsing errors
            {
                "args": [                                   # error object args
                    <value>,                                # first argument is usually an error message
                    ...                                     # repeats for each argument
                ],
                "word": <value>                             # word that could not be parsed
            },
            ...                                             # repeats for each error
        ],
        "_raw": <raw gsid value>,                           # copy of the original parsed value
        "revision": {                                       # only if any subkey is present
            "arc_merge": <value>,                           # value of `ARC_MERGE` key, if present
            "arcanum": <value as int>,                      # value of `ARCANUM` key, if present
            "svn": <value as int>,                          # value of `SVN` key, if present
            "svn_txn": <value>                              # value of `SVN_TXN` key, if present
        },
        "testenv": {                                        # only if `TE` key is present
            "db_name": <value>,
            "job_name": <value>,
            "job_id": <value as integer>,
            "circuit": <value>
        },
        "sandbox": [                                        # only if `SB` keys are present
            {
                "task_type": <value>,
                "task_id": <value as integer>,
                "user": <value>                             # if `USER` key followed `SB`
            },
            ...                                             # repeats for each SB key
        ],
        "svn_diff": [                                       # only if `SVN_DIFF` key is present
            <revision 1 as integer>,
            <revision 2 as integer>
        ],
        "user": <value>                                     # if `USER` key not followed `SB`
    }
    ```
    All values are strings.

    :param str text: text of GSID context attribute
    :param bool ignore_errors: if true, errors are returned in the `_errors` key instead of raising exception
    :return: tracing information in the format described above
    :rtype: dict
    :raises: GsidWordParseError if some GSID words could not be parsed and `ignore_errors` is false
    '''
    result = dict(_raw=text)
    errors = []
    for word in text.split():
        try:
            key, delimiter, data = word.partition(':')
            if not delimiter:
                raise GsidWordParseError('Missing delimiter')
            target = result
            if key in ('ARC_MERGE', 'SVN_TXN'):
                target = result.setdefault('revision', {})
                key = key.lower()
            elif key in ('ARCANUM', 'SVN'):
                target = result.setdefault('revision', {})
                key = key.lower()
                data = __gsid_int(data)
            elif key == 'SVNDIFF':
                key = 'svn_diff'
                data = __gsid_element_info(('revision1', 'revision2'), data)
                data = [__gsid_int(data[field]) for field in ('revision1', 'revision2')]
            elif key == 'TE':
                key = 'testenv'
                # Cf. https://a.yandex-team.ru/arc/trunk/arcadia/sandbox/projects/release_machine/core/const/__init__.py?rev=r9059522#L47
                data = __gsid_element_info(('db_name', 'job_name', 'job_id', 'circuit'), data)
                data['job_id'] = __gsid_int(data['job_id'])
            elif key == 'SB':
                data = __gsid_element_info(('task_type', 'task_id'), data)
                data['task_id'] = __gsid_int(data['task_id'])
                result.setdefault('sandbox', []).append(data)
                continue
            elif key == 'USER':
                if 'sandbox' in result:
                    target = result['sandbox'][-1]
                key = key.lower()
            else:
                raise GsidWordParseError('Unknown GSID key: {}'.format(key))
            if key in target:
                raise GsidWordParseError('Duplicate GSID key: {}'.format(key))
            target[key] = data
        except GsidWordParseError as parse_error:
            if ignore_errors:
                errors.append(dict(word=word, args=parse_error.args))
            else:
                raise
    if errors:
        result.update(_errors=errors)
    return result
