'''
Information about Python stack.
'''

from __future__ import absolute_import, division, print_function

import os.path
import six

from sandbox.projects.yabs.sandbox_task_tracing.defaults import DEFAULTS
from sandbox.projects.yabs.sandbox_task_tracing.util import frozendict


# Common file prefix for tracing library files to be excluded.
# Temporary value, can be changed by parent modules.
TRACING_FILES_PREFIX = os.path.dirname(__file__) + '/'


def set_tracing_files_prefix(prefix):
    '''
    Sets common prefix for all source file in tracing library.

    Usage:
    ```python
    set_tracing_files_prefix(os.path.dirname(__file__) + '/')
    ```
    '''
    global TRACING_FILES_PREFIX
    TRACING_FILES_PREFIX = prefix


def __frame_as_dict(frame_summary, __keys=('filename', 'lineno', 'name', 'line')):
    if type(frame_summary).__name__ == 'FrameSummary':  # newer version
        return {key: getattr(frame_summary, key) for key in __keys}
    else:
        return dict(zip(__keys, frame_summary))


def stack_info(frames, stack_info_spec=frozendict(), is_exception=False):
    '''
    Returns information about Python stack.

    Returned value after conversion to JSON:
    ```json
    {
        "last_frame": {                                     # only if `stack_info_spec['last_frame']`
            "filename": <file name>,                        # string
            "line": <source line>,                          # string, possibly stripped of whitespace
            "lineno": <line number>,                        # integer
            "name": <enclosing scope name>                  # string name of function or other kind of scope
        },
        "traceback": [                                      # only if `stack_info_spec['traceback']`
            {
                ...                                         # see description of `"last_frame"` above
            },
            ...                                             # repeats for each frame
        ]
    }
    ```

    :param list frames: as returned by `traceback.extract_stack` or `traceback.extract_tb` (element type depends on Python version)
    :param dict stack_info_spec:
        stack_info spec in the following format:
        ```json
        {
            'exclude_tracing_files': <bool>,                # exclude tracing library files from the result
            'excluded_files': <str | Container[str]>        # exclude specified file(s) when finding last frame
            'last_frame': <bool>,                           # include last frame in the result
            'traceback': <bool>                             # include traceback in the result
        }
        ```
        All fields are optional, see `..defaults.DEFAULTS['stack_info_spec']` for defaults.
    :param bool is_exception: indicates frames belong to exception
    :return: tracing information in the format described above
    :rtype: dict
    '''
    result = {}
    add_traceback = stack_info_spec.get('traceback', DEFAULTS['stack_info_spec']['traceback'])
    add_last_frame = stack_info_spec.get('last_frame', DEFAULTS['stack_info_spec']['last_frame'])
    if add_traceback or add_last_frame:
        frames_info = map(__frame_as_dict, frames)
        if stack_info_spec.get('exclude_tracing_files', DEFAULTS['stack_info_spec']['exclude_tracing_files']):
            frames_info = filter(lambda frame_info: not frame_info['filename'].startswith(TRACING_FILES_PREFIX), frames_info)
        frames_info = tuple(frames_info)

        if add_traceback:
            result.update(traceback=frames_info)

        if add_last_frame:
            if not is_exception:
                excluded_files = stack_info_spec.get('excluded_files', ())
                if isinstance(excluded_files, six.string_types):
                    excluded_files = {excluded_files}
                if excluded_files:
                    frames_info = tuple(filter(lambda frame_info: frame_info['filename'] not in excluded_files, frames_info))
            last_frame_info = frames_info[-1] if frames_info else None
            result.update(last_frame=last_frame_info)
    return result
