'''
Information about Sandbox task.
'''

from __future__ import absolute_import, division, print_function

import sandbox.sdk2

from sandbox.projects.yabs.sandbox_task_tracing.info.gsid import gsid_info
from sandbox.projects.yabs.sandbox_task_tracing.info.symbol import symbol_info


def task_info(task, format_='brief', __supported_formats={'brief', 'full'}):
    '''
    Returns information about Sandbox task object.

    Returned value after conversion to JSON, in brief format:
    ```json
    {
        "id": <task id>,                                    # integer
        "task_type": <sandbox task type>,                   # string, typically IN_THIS_FORMAT
        "type": <Python task object type info>              # as returned by `.symbol.symbol_info`
    }
    ```

    Returned value after conversion to JSON, in full format:
    ```json
    {
        "_raw": <task info>,                                # as returned by server
        "audit": <audit records>,                           # as returned by server, typically a list
        "gsid": <parsed gsid>,                              # as returned by `.gsid.gsid_info`
        ...                                                 # all items from brief format
    }
    ```

    :param task: task object
    :param Literal['brief', 'full'] format_: info format
    :return: tracing information in the format described above
    :rtype: dict
    '''
    if format_ not in __supported_formats:
        raise ValueError('Passed format {!r} does not belong to supported formats {}'.format(format_, __supported_formats))
    task_id = task.id
    server_task = sandbox.sdk2.Task.current.server.task[task_id]
    server_task_info = server_task.read()
    assert server_task_info['id'] == task_id
    result = dict(
        id=task_id,
        task_type=server_task_info['type'],
        type=symbol_info(type(task)),
    )
    if format_ == 'full':
        audit = server_task.audit.read()
        context = server_task.context.read()
        gsid = context.get('__GSID')
        result.update(
            _raw=server_task_info,
            audit=audit,
            gsid=(gsid_info(gsid) if gsid is not None else None),
        )
    return result
