'''
Common utility functions for sandbox task tracing library.
'''

from __future__ import absolute_import, division, print_function

import datetime
import functools
import itertools
import re
import six
import time


def coalesce(*args):
    '''
    Returns first non-`None` argument, `None` otherwise.

    :param list *args: arguments
    :return: first element of `args` that is not `None`, otherwise `None`
    '''
    for value in args:
        if value is not None:
            return value
    return None


def decorator_function(decorator_function):
    '''
    Decorator for decorator-returning functions.

    Provides two features:
    * Allows function to be used as decorator without arguments.
    * Fixes result name and makes it picklable.

    :param decorator_function: decorator-returning function
    :type decorator_function: Callable[P, Callable[Callable, Callable]]
    :return: enhanced decorator-returning function that can also be used directly as decorator
    :rtype Callable[Callable, Callable] | Callable[P, Callable[Callable, Callable]]
    '''

    @functools.wraps(decorator_function)
    def wrapper(*args, **kwargs):

        def naming_decorator(function):
            result = decorator(function)
            result._wrapped = function
            functools.update_wrapper(result, function)
            return result

        if len(args) == 1 and not kwargs and callable(args[0]):
            decorator = decorator_function()
            return naming_decorator(args[0])
        else:
            decorator = decorator_function(*args, **kwargs)
            return naming_decorator

    return wrapper


class frozendict(dict):
    '''
    Very simple immutable dict implementation.

    Does not provide perfect future-proof protection.
    Probably can be improved using __getattribute__ with whitelist,
    but current implementation is sufficient for the library internal use.
    '''

    def raise_error(self, *args, **kwargs):
        raise TypeError('This object is supposed to be immutable')

    __ior__ = raise_error
    __delitem__ = raise_error
    __setitem__ = raise_error
    clear = raise_error
    pop = raise_error
    popitem = raise_error
    setdefault = raise_error
    update = raise_error

    def __repr__(self):
        return '{}({})'.format(type(self).__name__, super(frozendict, self).__repr__())

    def __add__(self, other):
        '''
        Returns copy of the current dict updated with other dict.

        Keys of the `other` dict have priority.

        :param Mapping | Iterable other: any value suitable as first argument of `dict.update`
        :return frozendict: sum of `self` and `other` dictionaries
        '''
        result = frozendict(self)
        super(frozendict, result).update(other)
        return result

    def __radd__(self, other):
        '''
        Returns copy of the other dict updated with current dict.

        Keys of the `self` dict have priority.

        :param Mapping | Iterable other: any value suitable as first argument of `dict.__init__`
        :return frozendict: sum of `self` and `other` dictionaries
        '''
        result = frozendict(other)
        super(frozendict, result).update(self)
        return result

    def updated(self, *args, **kwargs):
        if len(args) > 1:
            raise TypeError('updated expected at most 1 argument, got {}'.format(len(args)))
        result = frozendict(self)
        for arg in args:
            super(frozendict, result).update(arg)
        super(frozendict, result).update(kwargs)
        return result

    __slots__ = ()


# using dateutil is difficult in Sandbox
def microseconds_from_utc_iso(
    text,
    __regex=re.compile(''.join((
        r'^',
        r'(?P<year>\d{4})-',
        r'(?P<month>\d{1,2})-',
        r'(?P<day>\d{1,2})[T ]',
        r'(?P<hour>\d{1,2}):',
        r'(?P<minute>\d{1,2}):',
        r'(?P<second>\d{1,2})',
        r'([.,](?P<microsecond>\d*))?Z',
        r'$',
    ))),
    __epoch_start=datetime.datetime(1970, 1, 1),
):
    '''
    Parses date/time in ISO 8601 format as used in sandbox.

    :param str text: date/time in ISO 8601 format (only a limited subset of cases is supported, see `__regex`).
    :return int: number of microseconds since unix epoch (google unix time for details)
    '''
    match = __regex.match(text)
    if not match:
        raise ValueError('Value {!r} does not conform to pattern {}'.format(text, __regex.pattern))
    groups = match.groupdict('')
    groups['microsecond'] = groups['microsecond'][:6].ljust(6, '0')
    timestamp = datetime.datetime(**{key: int(value) for key, value in six.iteritems(groups)})
    return int((timestamp - __epoch_start).total_seconds() * 1E6)


# using more_itertools is difficult in Sandbox
def pairwise(iterable):
    '''
    Implementation of `more_itertools.pairwise`.

    Source: https://docs.python.org/2/library/itertools.html#recipes
    '''
    a, b = itertools.tee(iterable)
    next(b, None)
    return six.moves.zip(a, b)


if hasattr(time, 'perf_counter_ns'):
    def time_microseconds(__reference=(time.time_ns() - time.perf_counter_ns())):
        '''
        Returns current time in microseconds.

        Uses integer perf_counter source, if available.
        '''
        return (__reference + time.perf_counter_ns()) // 1000
elif hasattr(time, 'perf_counter'):
    def time_microseconds(__reference=(time.time() - time.perf_counter())):
        '''
        Returns current time in microseconds.

        Uses floating-point perf_counter source, if available.
        '''
        return int((__reference + time.perf_counter()) * 1E6)
else:
    def time_microseconds():
        '''
        Returns current time in microseconds.

        Uses system clock (not monotonic, may go backwards).
        '''
        return int(time.time() * 1E6)
