'''
Sandbox task tracing library: wrapper for subprocess module.
'''

from __future__ import absolute_import, division, print_function

from sandbox.projects.yabs.sandbox_task_tracing import trace_subprocess_calls


class SubprocessWrapper(object):

    def _completed_process_class(outer_self):

        # cf. https://github.com/python/cpython/blob/2bde6827ea4f136297b2d882480b981ff26262b6/Lib/subprocess.py#L465
        class CompletedProcess(object):

            def __init__(self, args, returncode, stdout=None, stderr=None):
                self.args = args
                self.returncode = returncode
                self.stdout = stdout
                self.stderr = stderr

            def __repr__(self):
                args = ['args={!r}'.format(self.args), 'returncode={!r}'.format(self.returncode)]
                if self.stdout is not None:
                    args.append('stdout={!r}'.format(self.stdout))
                if self.stderr is not None:
                    args.append('stderr={!r}'.format(self.stderr))
                return "{}({})".format(type(self).__name__, ', '.join(args))

            def check_returncode(self):
                """Raise CalledProcessError if the exit code is non-zero."""
                if self.returncode:
                    # stderr argument requires version 3.5
                    raise outer_self.CalledProcessError(self.returncode, self.args, self.stdout)

        return CompletedProcess

    def _popen_and_communicate_function(self):

        @trace_subprocess_calls
        def popen_and_communicate(args, *other_args, **kwargs):
            '''
            Poor-man's `subprocess.run`.

            Creates new `Popen` object then runs its `communicate` method.

            Accepts all parameters that `subprocess.Popen` accepts
            plus `input` keyword parameter that is passed to `communicate`.

            :return CompletedProcess: completed process object
            '''
            input = kwargs.pop('input', None)
            process = self.Popen(args, *other_args, **kwargs)
            stdout, stderr = process.communicate(input=input)
            return self.CompletedProcess(args, process.returncode, stdout, stderr)

        return popen_and_communicate

    def _popen_and_wait_function(self):

        @trace_subprocess_calls
        def popen_and_wait(args, *other_args, **kwargs):
            '''
            Poor-man's `subprocess.run`.

            Creates new `Popen` object then runs its `wait` method.

            :return int: `Popen.wait` result which is return code
            '''
            return self.Popen(args, *other_args, **kwargs).wait()

        return popen_and_wait

    def __init__(self, subprocess):
        for name in dir(subprocess):
            if name.startswith('_'):
                continue
            value = getattr(subprocess, name)
            if name in {'call', 'check_call', 'check_output', 'run'}:
                assert callable(value), (name, value, type(value))
                value = trace_subprocess_calls(value)
            setattr(self, name, value)

        if not hasattr(self, 'CompletedProcess'):
            self.CompletedProcess = self._completed_process_class()

        self.popen_and_communicate = self._popen_and_communicate_function()
        self.popen_and_wait = self._popen_and_wait_function()

    def inject_into(self, target):
        for name in dir(self):
            if name.startswith('_'):
                continue
            setattr(target, name, getattr(self, name))
