from __future__ import absolute_import, division, print_function

from abc import abstractmethod

from sandbox.projects.yabs.sandbox_task_tracing.util import frozendict


class CallSequenceError(Exception):
    '''
    Raised if the writer has not been properly initialized before use.

    Specifically, `set_task` must be called once before any calls to `write_records`.
    '''
    pass


class AbstractTraceWriter(object):
    '''
    Trace writer abstract class.

    Writes trace records to the some final location.

    Thread-safety: does not have to be thread-safe.
    All methods including constructor are only ever called from a single thread.
    (Specifically, `buffer.TraceWritingThread`.)
    '''

    @abstractmethod
    def set_task(self, task_id, iteration_id, task_info=frozendict()):
        '''
        Sets task parameters in the writer.

        Must be called once before any calls to `write_records`.

        All calls to the same class must have distinct `(task_id, iteration_id)` pair.
        If this condition is violated, the behavior is undefined.

        :param int task_id: sandbox task id
        :param int iteration_id: sandbox task iteration id
        :param dict task_info: arbitrary task info, JSON-compatible
        '''
        pass

    @abstractmethod
    def write_records(self, records):
        '''
        Writes trace records.

        All records within the writer lifetime must have distinct id (`TraceRecord.record`).
        If this condition is violated, the behavior is undefined.

        :param Iterable[TraceRecord] records: records to write
        '''
        pass

    @abstractmethod
    def flush(self, timeout=None):
        '''
        Flushes writer buffers, if any.

        :param float timeout: timeout in seconds, class-dependent default if `None`
        '''
        pass
