from __future__ import absolute_import, division, print_function

from sandbox.projects.yabs.sandbox_task_tracing.util import frozendict

from sandbox.projects.yabs.sandbox_task_tracing.writers.abstract import AbstractTraceWriter, CallSequenceError


class YtTraceWriter(AbstractTraceWriter):
    '''
    Trace writer writing trace records to a pair of YT dynamic tables.

    Database schema is described in:
    https://wiki.yandex-team.ru/users/marat-khalili/trejjsing-v-shm/#sxemadannyx
    '''

    def __init__(self, proxy, path, token, config=frozendict()):
        '''
        Initializes new YT trace writer instance.

        :param str proxy: YT proxy to use
        :param str path: YT path with pre-existing dynamic tables
        :param str token: YT token
        '''
        try:
            import yt.wrapper
            yt.wrapper  # hide 'imported but unused' error
        except ImportError:
            from sandbox.sandboxsdk.environments import PipEnvironment
            PipEnvironment("yandex-yt", use_wheel=True).prepare()
        from yt.wrapper import JsonFormat, YtClient, ypath_join
        config = frozendict(
            tabular_data_format=JsonFormat(),
        ) + config
        self.client = YtClient(proxy=proxy, token=token, config=config)
        self.tasks_path = ypath_join(path, 'tasks')
        self.records_path = ypath_join(path, 'records')
        self.tasks_atomicity = self.client.get(ypath_join(self.tasks_path, '@atomicity'))
        self.records_atomicity = self.client.get(ypath_join(self.records_path, '@atomicity'))
        self.task_id = None
        self.iteration_id = None

    def set_task(self, task_id, iteration_id, task_info=frozendict()):
        if self.task_id is not None or self.iteration_id is not None:
            raise CallSequenceError('Method `set_task` should only be called once')

        assert task_id is not None
        assert iteration_id is not None

        inserted_rows = (dict(task=task_id, info=task_info),)
        self.client.insert_rows(self.tasks_path, inserted_rows, atomicity=self.tasks_atomicity)

        self.task_id = task_id
        self.iteration_id = iteration_id

    def __record_as_row(self, record):
        result = record._asdict()
        result.update(task=self.task_id, iteration=self.iteration_id)
        return result

    def write_records(self, records):
        if self.task_id is None or self.iteration_id is None:
            raise CallSequenceError('Method `set_task` must be called first')
        inserted_rows = map(self.__record_as_row, records)
        self.client.insert_rows(self.records_path, inserted_rows, atomicity=self.records_atomicity)

    def flush(self, timeout=None):
        # this writer is synchronous
        pass
