import logging
from datetime import datetime
import time

from sandbox import sdk2
from sandbox.projects.yabs.sbyt.lib.solomon_client import YabsStatSolomonClientBase
from sandbox.sandboxsdk import environments

LOGGER = logging.getLogger(__name__)


class AppsflyerSendCostsMonitoring(sdk2.Task):
    """
        Appsflyer send costs errors monitoring
    """

    class Parameters(sdk2.Task.Parameters):
        description = 'Appsflyer send costs monitoring'

        yql_token_secret = sdk2.parameters.YavSecret(
            'Yav secret with YQL token',
            required=True,
        )

        solomon_token_secret = sdk2.parameters.YavSecret(
            'Yav secret with Solomon token',
            required=True,
        )

        with sdk2.parameters.RadioGroup('Solomon instance') as solomon_instance:
            solomon_instance.values['preprod'] = solomon_instance.Value('Prestable', default=True)
            solomon_instance.values['prod'] = solomon_instance.Value('Production')

    class Requirements(sdk2.Task.Requirements):
        environments = (
            environments.PipEnvironment('yandex-yt', version='0.8.38a1'),
            environments.PipEnvironment('yandex-yt-yson-bindings-skynet'),
            environments.PipEnvironment('yql', version='1.2.91'),
            environments.PipEnvironment('solomon')
        )
        cores = 1
        disk_space = 1 * 1024

    def _create_yql_client(self):
        from yql.api.v1.client import YqlClient

        token = self.Parameters.yql_token_secret.data()['yt_token']
        return YqlClient(token=token)

    def _send_metrics(self, cluster, stat):
        solomon_token = self.Parameters.solomon_token_secret.data()['solomon_token']
        s_cli = YabsStatSolomonClientBase(
            "yabs_stat_sbadan",
            cluster="yabs_stat_sandbox",
            is_production=self.Parameters.solomon_instance == 'prod',
            token=solomon_token,
        )
        for row in stat:
            labels = {'metric': 'Number of sendings'}
            sensors = [
                'NumberOfSends',
                'SuccessSends',
                'FailedSends',
                'FailedCheckHttpError',
                'FailedValidation',
                'FailedAll',
                'FailPercent',
            ]
            values = []
            for s in sensors:
                values.append(int(row[s]))

            s_cli.push_metrics(
                sensors,
                values,
                [labels] * len(sensors),
                datetime.utcfromtimestamp(time.time())
            )

    YQL_OPERATION_PATTERN = 'https://yql.yandex-team.ru/Operations/{id}'

    def _get_share_url(self, operation):
        from yql.client.operation import YqlOperationShareIdRequest

        share_request = YqlOperationShareIdRequest(operation.operation_id)
        share_request.run()

        return self.YQL_OPERATION_PATTERN.format(id=share_request.json)

    def _yql_operation_callback(self, operation):
        self.operation_id = operation.operation_id
        share_url = self._get_share_url(operation)
        LOGGER.info('YQL shared link : %s', share_url)

    def on_execute(self):
        yql = self._create_yql_client()
        clusters = [
            'hahn',
        ]

        query_text = "PRAGMA yt.InferSchema = '1';"

        for cluster in clusters:
            query_text = query_text + """

            use {};
            $last_table = (
                SELECT
                    MAX(Path) AS last_table
                FROM
                    FOLDER('//home/yabs/stat/offline/appsflyer-send-costs/production/send_to_appsflyer', "")
                WHERE
                    Type = "table"
            );

            $last_table_check = (
                SELECT
                    MAX(Path) AS last_table
                FROM
                    FOLDER('//home/yabs/stat/offline/appsflyer-send-costs/production/check_status', "")
                WHERE
                    Type = "table"
            );

            SELECT
                SUM(NumberOfSends) AS NumberOfSends,
                SUM(SuccessSends) AS SuccessSends,
                SUM(FailedSends) AS FailedSends,
                SUM(FailedCheckHttpError) AS FailedCheckHttpError,
                SUM(FailedValidation) AS FailedValidation,
                SUM(FailedSends) + SUM(FailedCheckHttpError) + SUM(FailedValidation) AS FailedAll,
                100.0 * (SUM(FailedSends) + SUM(FailedCheckHttpError) + SUM(FailedValidation)) / SUM(NumberOfSends) AS FailPercent
            from (
                SELECT
                    COUNT_IF(code != 200) AS FailedSends,
                    COUNT(*) AS NumberOfSends,
                    0 AS FailedCheckHttpError,
                    0 AS FailedValidation
                FROM
                    CONCAT($last_table)

                UNION ALL

                SELECT
                    0 AS FailedSends,
                    0 AS NumberOfSends,
                    COUNT_IF(check_code == 200 AND status == 'Applied') AS SuccessSends,
                    COUNT_IF(check_code != 200) AS FailedCheckHttpError,
                    COUNT_IF(status != 'Applied') AS FailedValidation
                FROM
                    CONCAT($last_table_check)
            );
""".format(cluster)

        query = yql.query(query_text, syntax_version=1)
        query.run(pre_start_callback=self._yql_operation_callback)
        results = query.get_results()

        if not results.is_success:
            msg = '\n'.join([str(err) for err in results.errors])
            LOGGER.error('Error when executing query: %s', msg)
            raise Exception(msg)
        else:
            LOGGER.info('Query successfully finished. Collecting result')

        for (cluster, table) in zip(clusters, results):
            result = []
            table.fetch_full_data()
            columns = []
            for column_name, column_type in table.columns:
                columns.append(column_name)
            for row in table.rows:
                result.append(dict([(columns[i], value) for i, value in enumerate(row)]))

            self._send_metrics(cluster, result)
