"""Yabs SBYT linker."""
import logging

from sandbox import sdk2

from sandbox.common.types import client as ctc
from sandbox.sandboxsdk import environments
from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk import task

from sandbox.projects.common import utils
from sandbox.projects.common.yabs import ytstat

import datetime


LINKER_CONFIG = {
    'hahn': {
        '//cooked_logs/bs-chevent-cooked-log/1d': {
            'start_time': '2021-04-20',
            'relink_on_modification': False,
            'delay': 2 * 3600,
        },
        '//logs/bs-chevent-log/1h': {
            'start_time': '2017-10-26T00:00:00',
        },
        '//home/antifraud/logs/bs-chhit-log/1h': {
            'start_time': '2017-10-26T10:00:00',
        },
        '//home/antifraud/logs/bs-chhit-log/1d': {
            'start_time': '2021-04-20',
            'relink_on_modification': True,
            'delay': 6 * 3600,
        },
        '//logs/bs-dsp-checked-log/1h': {
            'start_time': '2019-05-23',
        },
        '//logs/bs-rtb-log/1h': {
            'start_time': '2018-10-08',
        },
        '//logs/bs-mobile-postclick-log/stream/5min': {
            'log_type': 'bs-mobile-postclick-log',
            'start_time': '2019-09-05T19:00:00',
        },
        '//logs/bs-uniform-postback-log/stream/5min': {
            'log_type': 'bs-uniform-postback-log',
            'start_time': '2020-06-18T15:00:00',
        },
        '//logs/visit-v2-ads-log/stream/5min' : {
            'log_type': 'visit-v2-ads-log',
            'start_time': '2019-08-01T16:50:00',
        },
        '//home/yabs/stat/offline/qtail-joined-log/production/hahn/joined-qtail-buffer-log': {
            'log_type': 'offline_joined-qtail-buffer-log',
            'start_time': '2021-04-29T00:00:00',
        },
        '//logs/adfox-event-log/stream/1min': {
            'log_type': 'adfox-event-log',
            'start_time': '2020-06-29T12:30:00',
        },
        '//logs/adfox-undoevent-log/stream/1min': {
            'log_type': 'adfox-undoevent-log',
            'start_time': '2021-11-30T10:10:00',
        },
        '//logs/adfox-joined-log/stream/1min': {
            'log_type': 'adfox-joined-log',
            'start_time': '2022-05-12T16:00:00',
        },
        '//logs/adfox-ui-external-stat/stream/5min': {
            'log_type': 'adfox-external-stat-log',
            'start_time': '2022-05-17T00:00:00',
        },
        '//logs/adfox-ui-external-stat-raw/stream/5min': {
            'log_type': 'adfox-external-stat-log-raw',
            'start_time': '2022-06-21T00:00:00',
        },
        '//logs/bs-proto-newshit-log/stream/5min': {
            'log_type': 'bs-proto-newshit-log',
            'start_time': '2021-12-14T00:00:00',
        },
        '//logs/bs-proto-dspcreative-log/30min': {
            'log_type': 'bs-proto-dspcreative-log',
            'start_time': '2020-09-01T12:00:00',
        },
        '//logs/bs-dsp-creative-moderation-response-log/1h': {
            'log_type': 'bs-dsp-creative-moderation-response-log',
            'start_time': '2020-09-10T16:00:00',
        },
        '//logs/appmetrica-ecommerce-log/stream/5min': {
            'log_type': 'appmetrica-ecommerce-log',
            'start_time': '2020-10-07T10:00:00',
        },
        '//logs/bs-proto-mismatchedpatterns-log/1h': {
            'log_type': 'bs-proto-mismatchedpatterns-log',
            'start_time': '2021-06-25T16:00:00',
        },
        '//home/balance/prod/log_tariff/public/expense/partner/tariffed': {
            'log_type': 'billing-tariffed',
            'start_time': '2021-02-26T17:00:00',
        },
        '//home/balance/prod/log_tariff/public/expense/act_partner/retariffed_events': {
            'log_type': 'billing-tariffed',
            'start_time': '2021-09-01_0001',
            'table_name_to_datetime': lambda table_name: datetime.datetime.strptime(table_name.split('_')[0], "%Y-%m-%d"),
        },
        '//home/balance/prod/log_tariff/public/expense/partner/tariffed_internal': {
            'log_type': 'billing-tariffed-internal',
            'start_time': '2021-02-26T17:00:00',
        },
        '//home/balance/prod/log_tariff/public/income/dsp/tariffed': {
            'log_type': 'billing-tariffed-dsp',
            'start_time': '2022-03-01T00:00:00',
        },
        '//logs/appmetrica-skadnetwork-log/stream/5min': {
            'log_type': 'appmetrica-skadnetwork-log',
            'start_time': '2021-04-14T15:00:00',
        },
        '//logs/bs-skadnetwork-enriched-log/stream/5min': {
            'log_type': 'bs-skadnetwork-enriched-log',
            'start_time': '2021-04-16T19:00:00',
        },
        '//logs/appmetrica-external-events/stream/5min': {
            'log_type': 'appmetrica-external-events',
            'start_time': '2021-04-28T00:00:00',
        },
        '//logs/appmetrica-yandex-events/stream/5min': {
            'log_type': 'appmetrica-yandex-events',
            'start_time': '2021-04-28T00:00:00',
        },
        '//logs/superapp-metrika-mobile-log/stream/5min': {
            'log_type': 'superapp-metrika-mobile-log',
            'start_time': '2020-01-28T00:00:00',
        },
        '//logs/browser-metrika-mobile-log/stream/5min': {
            'log_type': 'browser-metrika-mobile-log',
            'start_time': '2021-04-28T00:00:00',
        },
        '//logs/disk-metrika-mobile-log/stream/5min': {
            'log_type': 'disk-metrika-mobile-log',
            'start_time': '2021-04-28T00:00:00',
        },
        '//logs/navi-metrika-mobile-log/stream/5min': {
            'log_type': 'navi-metrika-mobile-log',
            'start_time': '2021-04-28T00:00:00',
        },
        '//logs/taxi-metrika-mobile-log/stream/5min': {
            'log_type': 'taxi-metrika-mobile-log',
            'start_time': '2021-04-28T00:00:00',
        },
        '//logs/bs-proto-candidates-log/stream/5min': {
            'log_type': 'bs-proto-candidates-log',
            'start_time': '2022-02-17T00:00:00',
        },
        '//logs/appmetrica-aggregated-uniform-postback-log/stream/5min': {
            'log_type': 'appmetrica-aggregated-uniform-postback-log',
            'start_time': '2022-03-04T14:00:00',
        },
    },
    'arnold': {
        '//cooked_logs/bs-chevent-cooked-log/1d': {
            'start_time': '2021-04-20',
            'relink_on_modification': False,
            'delay': 2 * 3600,
        },
        '//logs/bs-chevent-log/1h': {
            'start_time': '2018-10-26T00:00:00',
        },
        '//home/antifraud/logs/bs-chhit-log/1h': {
            'start_time': '2018-10-26T10:00:00',
        },
        '//home/antifraud/logs/bs-chhit-log/1d': {
            'start_time': '2021-04-20',
            'relink_on_modification': True,
            'delay': 6 * 3600,
        },
        '//logs/bs-dsp-checked-log/1h': {
            'start_time': '2019-05-23',
        },
        '//logs/bs-rtb-log/1h': {
            'start_time': '2018-10-08',
        },
        '//logs/bs-mobile-postclick-log/stream/5min': {
            'log_type': 'bs-mobile-postclick-log',
            'start_time': '2019-09-05T19:00:00',
        },
        '//logs/bs-dsp-log/stream/5min': {
            'log_type': 'bs-dsp-log',
            'start_time': '2022-06-21T17:30:00',
        },
        '//logs/bs-dsp-log/1d': {
            'log_type': 'bs-dsp-day-log',
            'start_time': '2022-07-21',
            'delay': 86400,
        },
        '//logs/bs-dsp-checked-log/1d': {
            'log_type': 'bs-dsp-checked-day-log',
            'start_time': '2022-07-21',
            'delay': 86400,
        },
        '//logs/visit-v2-ads-log/stream/5min' : {
            'log_type': 'visit-v2-ads-log',
            'start_time': '2019-09-17T12:00:00',
        },
        '//home/yabs/stat/offline/qtail-joined-log/production/arnold/joined-qtail-buffer-log': {
            'log_type': 'offline_joined-qtail-buffer-log',
            'start_time': '2021-04-29T00:00:00',
        },
        '//logs/adfox-event-log/stream/1min': {
            'log_type': 'adfox-event-log',
            'start_time': '2020-06-23T15:30:00',
        },
        '//logs/adfox-undoevent-log/stream/1min': {
            'log_type': 'adfox-undoevent-log',
            'start_time': '2021-11-30T10:10:00',
        },
        '//logs/adfox-ui-external-stat/stream/5min': {
            'log_type': 'adfox-external-stat-log',
            'start_time': '2022-04-19T15:00:00',
        },
        '//logs/adfox-ui-external-stat-raw/stream/5min': {
            'log_type': 'adfox-external-stat-log-raw',
            'start_time': '2022-06-21T00:00:00',
        },
        '//logs/adfox-joined-log/stream/1min': {
            'log_type': 'adfox-joined-log',
            'start_time': '2022-05-12T16:00:00',
        },
        '//logs/bs-proto-newshit-log/stream/5min': {
            'log_type': 'bs-proto-newshit-log',
            'start_time': '2021-12-15T10:00:00',
        },
        '//logs/bs-proto-dspcreative-log/30min': {
            'log_type': 'bs-proto-dspcreative-log',
            'start_time': '2020-09-01T12:00:00',
        },
        '//logs/bs-dsp-creative-moderation-response-log/1h': {
            'log_type': 'bs-dsp-creative-moderation-response-log',
            'start_time': '2020-09-10T16:00:00',
        },
        '//logs/bs-proto-mismatchedpatterns-log/1h': {
            'log_type': 'bs-proto-mismatchedpatterns-log',
            'start_time': '2021-06-25T16:00:00',
        },
        '//logs/bs-uniform-postback-log/stream/5min': {
            'log_type': 'bs-uniform-postback-log',
            'start_time': '2020-12-02T17:00:00',
        },
        '//home/balance/prod/log_tariff/public/expense/partner/tariffed': {
            'log_type': 'billing-tariffed',
            'start_time': '2021-02-26T17:00:00',
        },
        '//home/balance/prod/log_tariff/public/expense/act_partner/retariffed_events': {
            'log_type': 'billing-tariffed',
            'start_time': '2021-09-01_0001',
            'table_name_to_datetime': lambda table_name: datetime.datetime.strptime(table_name.split('_')[0], "%Y-%m-%d"),
        },
        '//home/balance/prod/log_tariff/public/expense/partner/tariffed_internal': {
            'log_type': 'billing-tariffed-internal',
            'start_time': '2021-02-26T17:00:00',
        },
        '//home/balance/prod/log_tariff/public/income/dsp/tariffed': {
            'log_type': 'billing-tariffed-dsp',
            'start_time': '2022-03-01T00:00:00',
        },
        '//logs/appmetrica-external-events/stream/5min': {
            'log_type': 'appmetrica-external-events',
            'start_time': '2021-04-28T00:00:00',
        },
        '//logs/appmetrica-yandex-events/stream/5min': {
            'log_type': 'appmetrica-yandex-events',
            'start_time': '2021-04-28T00:00:00',
        },
        '//logs/superapp-metrika-mobile-log/stream/5min': {
            'log_type': 'superapp-metrika-mobile-log',
            'start_time': '2020-01-28T00:00:00',
        },
        '//logs/browser-metrika-mobile-log/stream/5min': {
            'log_type': 'browser-metrika-mobile-log',
            'start_time': '2021-04-28T00:00:00',
        },
        '//logs/disk-metrika-mobile-log/stream/5min': {
            'log_type': 'disk-metrika-mobile-log',
            'start_time': '2021-04-28T00:00:00',
        },
        '//logs/navi-metrika-mobile-log/stream/5min': {
            'log_type': 'navi-metrika-mobile-log',
            'start_time': '2021-04-28T00:00:00',
        },
        '//logs/taxi-metrika-mobile-log/stream/5min': {
            'log_type': 'taxi-metrika-mobile-log',
            'start_time': '2021-04-28T00:00:00',
        },
        '//logs/bs-skadnetwork-enriched-log/stream/5min': {
            'log_type': 'bs-skadnetwork-enriched-log',
            'start_time': '2021-04-27T00:00:00',
        },
        '//logs/bs-proto-candidates-log/stream/5min': {
            'log_type': 'bs-proto-candidates-log',
            'start_time': '2022-03-03T18:00:00',
        },
    },
    'freud': {
        '//cooked_logs/bs-chevent-cooked-log/1d': {
            'start_time': '2021-04-20',
            'relink_on_modification': False,
            'delay': 86400 + 3 * 3600,
        },
        '//logs/bs-chevent-log/1h': {
            'start_time': '2017-10-11T12:00:00',
        },
        '//logs/bs-mobile-postclick-log/stream/5min': {
            'log_type': 'bs-mobile-postclick-log',
            'start_time': '2019-09-05T19:00:00',
        },
        '//logs/bs-proto-mismatchedpatterns-log/1h': {
            'log_type': 'bs-proto-mismatchedpatterns-log',
            'start_time': '2021-06-25T16:00:00',
        },
        '//logs/bs-uniform-postback-prestable-log/stream/5min': {
            'log_type': 'bs-uniform-postback-log',
            'start_time': '2020-06-20T00:00:00',
        },
        '//logs/visit-v2-ads-log/stream/5min': {
            'log_type': 'visit-v2-ads-log',
            'start_time': '2019-10-03T12:35:00',
        },
        '//home/yabs/stat/offline/qtail-joined-log/prestable/freud/joined-qtail-buffer-log': {
            'log_type': 'offline_joined-qtail-buffer-log',
            'start_time': '2021-04-29T12:00:00',
        },
        '//logs/adfox-event-log/stream/5min': {
            'log_type': 'adfox-event-log',
            'start_time': '2022-04-26T00:00:00',
        },
        '//logs/adfox-undoevent-log/stream/5min': {
            'log_type': 'adfox-undoevent-log',
            'start_time': '2022-04-26T00:00:00',
        },
        '//logs/adfox-ui-external-stat/stream/5min': {
            'log_type': 'adfox-external-stat-log',
            'start_time': '2022-04-18T15:00:00',
        },
        '//logs/adfox-ui-external-stat-raw/stream/5min': {
            'log_type': 'adfox-external-stat-log-raw',
            'start_time': '2022-06-20T00:00:00',
        },
        '//logs/adfox-joined-log/stream/5min': {
            'log_type': 'adfox-joined-log',
            'start_time': '2022-05-12T16:00:00',
        },
        '//logs/bs-proto-dspcreative-log/30min': {
            'log_type': 'bs-proto-dspcreative-log',
            'start_time': '2020-09-02T12:00:00',
        },
        '//logs/appmetrica-ecommerce-log/stream/5min': {
            'log_type': 'appmetrica-ecommerce-log',
            'start_time': '2020-09-28T00:00:00',
        },
        '//logs/bs-proto-newshit-log/stream/5min': {
            'log_type': 'bs-proto-newshit-log',
            'start_time': '2021-12-14T00:00:00',
        },
        '//logs/bs-skadnetwork-enriched-log/stream/5min': {
            'log_type': 'bs-skadnetwork-enriched-log',
            'start_time': '2021-05-12T17:35:00',
        },
        '//home/yabs/log/billing/prod/tariffed': {
            'log_type': 'billing-tariffed',
            'start_time': '2021-05-17T14:00:00',
        },
        '//home/yabs/log/billing/prod/tariffed_internal': {
            'log_type': 'billing-tariffed-internal',
            'start_time': '2021-05-17T14:00:00',
        },
        '//home/yabs/log/billing/prod/retariffed': {
            'log_type': 'billing-tariffed',
            'start_time': '2021-05-17',
        },
        '//logs/appmetrica-skadnetwork-log/stream/5min': {
            'log_type': 'appmetrica-skadnetwork-log',
            'start_time': '2021-05-20T18:00:00',
        },
        '//logs/appmetrica-aggregated-uniform-postback-log/stream/5min': {
            'log_type': 'appmetrica-aggregated-uniform-postback-log',
            'start_time': '2021-12-29T15:00:00',
        },
        '//logs/bs-proto-candidates-log/stream/5min': {
            'log_type': 'bs-proto-candidates-log',
            'start_time': '2022-02-17T00:00:00',
        },
    },
}
LOG_FILE = 'linker.log'


class YtLogPath(parameters.SandboxStringParameter):
    name = 'yt_log_path'
    description = 'YT log path'
    default_value = '//home/yabs/log/yabs-rt'


class YtProxy(parameters.SandboxStringParameter):
    name = 'yt_proxy'
    description = 'YT proxy'
    required = True


class YtTokenVaultName(parameters.SandboxStringParameter):
    name = 'yt_token_vault_name'
    description = 'Token name for vault / pair "sec-id:key_name" for yav'


class YabsSbytLinker(task.SandboxTask):
    """Linker gives possibility to work with non-realtime (offline) logs using SBYT.

    Main role of this task is to create links from various logs into common SBYT logs directory,
    in order to SBYT's coordinator could process them as usual logs.
    """
    client_tags = ctc.Tag.LINUX_PRECISE
    cores = 1
    environment = (
        environments.PipEnvironment('yandex-yt', '0.8.38a1', use_wheel=True),
        environments.PipEnvironment('yandex-yt-yson-bindings-skynet', use_wheel=True),
    )
    input_parameters = (
        YtLogPath,
        YtProxy,
        YtTokenVaultName,
    )
    type = 'YABS_SBYT_LINKER'

    def on_execute(self):
        ytstat.prepare_log_file(self, LOG_FILE)
        with ytstat.stderr_redirect(LOG_FILE):
            self._do_on_execute()

    def _do_on_execute(self):
        yt_log_path = utils.get_or_default(self.ctx, YtLogPath)
        yt_proxy = utils.get_or_default(self.ctx, YtProxy)
        yt_secret_name = utils.get_or_default(self.ctx, YtTokenVaultName)

        if yt_secret_name:
            split = yt_secret_name.split(":")
            if len(split) == 1:
                logging.info("using vault token")
                yt_token = self.get_vault_data('YABS-YTSTAT', split[0])
            elif len(split) == 2:
                logging.info("using yav token")
                secret = sdk2.yav.Secret(split[0])
                yt_token = secret.data()[split[1]]
            else:
                yt_token = None
        else:
            yt_token = None

        ytc = ytstat.get_yt_client(yt_proxy, yt_token)
        ytstat.install_supervisor(self, ytc)

        from supervisor import linker
        linker.execute(ytc, yt_log_path, LINKER_CONFIG[yt_proxy.split('.', 1)[0]])


__Task__ = YabsSbytLinker
