import copy
import json
import logging
import math
import time
from datetime import datetime

from sandbox import sdk2
from sandbox.projects.yabs.sbyt.lib.solomon_client import YabsStatSolomonClientBase
from sandbox.sandboxsdk import environments

LOGGER = logging.getLogger(__name__)

CIRCUIT_PARAMETERS = {
    'fast': {
        'resource_type': 'FAST_MYSQL_TO_YT_APPLIER_CONFIG',
        'config_name': 'fast-mysql-to-yt-applier-config.json',
        'yt_proxy': 'markov',
    },
    'slow': {
        'resource_type': 'YABS_MYSQL_TO_YT_APPLIER_CONFIG',
        'config_name': 'mysql-to-yt-applier-config.json',
        'yt_proxy': 'markov',
    },
    'preprod': {
        'resource_type': 'PREPROD_MYSQL_TO_YT_APPLIER_CONFIG',
        'config_name': 'preprod-mysql-to-yt-applier-config.json',
        'yt_proxy': 'pythia',
    },
}

CIRCUIT_TO_SOLOMON_CLUSTER = {
    'fast': 'yabs_bsdb_to_yt_fast',
    'slow': 'yabs_bsdb_to_yt_slow',
    'preprod': 'yabs_bsdb_to_yt_pre',
}

LOCK_PATH = 'yt_lock_path'
TABLES = 'tables'

REQUIRED_FIELDS = [
    LOCK_PATH,
    TABLES,
]


class ReplicatorConfig(object):
    def __init__(self, config_dict):
        self._config = config_dict
        self._validate()

    @classmethod
    def from_json(cls, path_to_json_config):
        LOGGER.info('Loading config from %r', path_to_json_config)
        with open(path_to_json_config) as config_file:
            config_dict = json.load(config_file)
        return cls(config_dict)

    def _validate(self):
        for field in REQUIRED_FIELDS:
            if field not in self._config:
                raise ValueError('`{}` field is missing in config'.format(field))

    @property
    def lock_path(self):
        return self._config[LOCK_PATH]

    @property
    def tables(self):
        return self._config[TABLES]


class YabsMysqlToYtAuditorMonitoring(sdk2.Task):
    """
        YabsMysqlToYtAuditorMonitoring monitoring
    """

    class Parameters(sdk2.Task.Parameters):
        description = 'MySQL-to-YT auditor monitoring'

        yt_token_secret = sdk2.parameters.YavSecret(
            'Yav secret with YT token',
            required=True,
        )
        solomon_token_secret = sdk2.parameters.YavSecret(
            'Yav secret with Solomon token',
            required=True,
        )

        with sdk2.parameters.RadioGroup('Solomon instance') as solomon_instance:
            solomon_instance.values['preprod'] = solomon_instance.Value('Prestable', default=True)
            solomon_instance.values['prod'] = solomon_instance.Value('Production')

    class Requirements(sdk2.Task.Requirements):
        environments = (
            environments.PipEnvironment('yandex-yt', version='0.8.38a1'),
            environments.PipEnvironment('yandex-yt-yson-bindings-skynet'),
            environments.PipEnvironment('solomon')
        )

    class AuditorSolomonClient(YabsStatSolomonClientBase):
        DEF_LABELS = {
            'metric': 'TimeFromLastAction'
        }

        def __init__(self, cluster, is_production=False, token=None):
            YabsStatSolomonClientBase.__init__(self,
                                               "mysql_to_yt_auditor",
                                               CIRCUIT_TO_SOLOMON_CLUSTER[cluster],
                                               is_production,
                                               token)

    def _create_yt_client(self, cluster):
        import yt.wrapper as yt

        token = self.Parameters.yt_token_secret.data()['yt_token']
        return yt.YtClient(config={"token": token, "proxy": {"url": cluster}})

    def _send_metrics(self, cluster, stat):
        solomon_token = self.Parameters.solomon_token_secret.data()['solomon_token']
        s_cli = YabsMysqlToYtAuditorMonitoring.AuditorSolomonClient(
            cluster=cluster,
            is_production=self.Parameters.solomon_instance == 'prod',
            token=solomon_token,
        )
        labels = []
        sensors = []
        values = []
        unixts = long(time.time())

        for table, data in stat.iteritems():
            for sensor_name, sensor_val in data.iteritems():
                if sensor_val:
                    lb = dict(YabsMysqlToYtAuditorMonitoring.AuditorSolomonClient.DEF_LABELS)
                    lb['TableName'] = table
                    labels.append(lb)
                    sensors.append(sensor_name)
                    values.append(unixts - sensor_val)

        for i in range(int(math.ceil(len(values) / 10.))):
            s_cli.push_metrics(
                sensors[i * 10:(i + 1) * 10],
                values[i * 10:(i + 1) * 10],
                labels[i * 10:(i + 1) * 10],
                datetime.utcfromtimestamp(unixts)
            )

    #        s_cli.push_metrics(
    #            sensors,
    #            values,
    #            labels,
    #            datetime.datetime.fromtimestamp(unixts)
    #        )

    def on_execute(self):
        attrs = [
            'audit_passed',
            'audit_failed'
        ]

        for circuit_name, circuit in CIRCUIT_PARAMETERS.iteritems():
            cluster = circuit['yt_proxy']
            ytc = self._create_yt_client(cluster)

            config_resource_type = circuit['resource_type']
            LOGGER.info('Loading latest %r resource', config_resource_type)
            config_resource = sdk2.Resource[config_resource_type].find(
                attrs={'released': 'stable'},
            ).first()
            config_path = str(sdk2.ResourceData(config_resource).path)
            LOGGER.info('Resource %r loaded to path %r', config_resource.id, config_path)

            rpl_config = ReplicatorConfig.from_json(config_path)

            conf_tables = []
            for table_name, table_config in rpl_config.tables.iteritems():
                if table_config.get('mysql_shard_count'):
                    shards = int(table_config.get('mysql_shard_count'))
                    exclude = [0]
                    if table_config.get('skip_shards_for_audit_list'):
                        exclude + list(table_config.get('skip_shards_for_audit_list'))
                    conf_tables = conf_tables + map(lambda y: table_name[7:] + str(y),
                                                    list(filter(lambda x: x not in [0, 1], range(shards))))
                else:
                    conf_tables.append(table_name[7:])

            results = dict(zip(conf_tables,
                               map(lambda z: copy.copy(z),
                                   ([dict(zip(attrs, map(lambda x: None, attrs)))]) * len(conf_tables))
                               ))

            for table, params in ytc.get("//home/yabs/dict", attributes=attrs + ['type']).iteritems():
                if params.attributes['type'] != 'replicated_table':
                    continue

                for attr_name in attrs:
                    attr_value = params.attributes.get(attr_name)
                    if isinstance(attr_value, dict):
                        for shard, data in attr_value.iteritems():
                            table_name = table + str(shard)
                            if results.get(table_name):
                                results[table_name][attr_name] = long(data)
                    elif attr_value is not None:
                        if results.get(table):
                            results[table][attr_name] = long(attr_value)

            self._send_metrics(circuit_name, results)
