import logging
from datetime import datetime

from sandbox import sdk2
from sandbox.projects.yabs.base_bin_task import BaseBinTask

LOGGER = logging.getLogger(__name__)

CIRCUIT_PARAMETERS = {
    'fast': {
        'resource_type': 'FAST_MYSQL_TO_YT_APPLIER_CONFIG',
        'config_name': 'fast-mysql-to-yt-applier-config.json',
        'yt_proxy': 'markov',
    },
    'slow': {
        'resource_type': 'YABS_MYSQL_TO_YT_APPLIER_CONFIG',
        'config_name': 'mysql-to-yt-applier-config.json',
        'yt_proxy': 'markov',
    },
    'preprod': {
        'resource_type': 'PREPROD_MYSQL_TO_YT_APPLIER_CONFIG',
        'config_name': 'preprod-mysql-to-yt-applier-config.json',
        'yt_proxy': 'pythia',
    },
}


class YabsMysqlToYtRplMonitoring(BaseBinTask):
    """MySQL-to-YT dictionaries replication monitoring"""

    class Requirements(sdk2.Requirements):
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(BaseBinTask.Parameters):
        description = 'MySQL-to-YT tables replication monitoring'

        with BaseBinTask.Parameters.version_and_task_resource() as version_and_task_resource:
            resource_attrs = sdk2.parameters.Dict(
                'Filter resource by',
                default={'target': 'yabs/sbyt/mysql_to_yt_rpl_monitoring'}
            )

        with sdk2.parameters.Group('Credentials') as credentials:
            yt_token_secret = sdk2.parameters.YavSecret(
                'Yav secret with YT token',
                required=True,
            )
            bsdb_credentials_secret = sdk2.parameters.YavSecret(
                'Yav secret with BSDB credentials',
                required=True,
            )

        with sdk2.parameters.Group('Monitoring settings') as monitoring_settings:
            with sdk2.parameters.String('Replication circuit', required=True, default='preprod') as circuit:
                for name in CIRCUIT_PARAMETERS:
                    circuit.values[name] = name.capitalize()

            should_push_to_solomon = sdk2.parameters.Bool('Push results to Solomon', default=False)
            with should_push_to_solomon.value[True]:
                solomon_token_secret = sdk2.parameters.YavSecret(
                    'Yav secret with Solomon token',
                    required=True,
                )
                with sdk2.parameters.RadioGroup('Solomon instance') as solomon_instance:
                    solomon_instance.values['preprod'] = solomon_instance.Value('Prestable', default=True)
                    solomon_instance.values['prod'] = solomon_instance.Value('Production')

    def on_execute(self):
        from yabs.sbyt.mysql_to_yt_rpl.monitoring.lib import (
            ReplicatorConfig,
            ReplicaMonitoring,
            SolomonClient,
        )

        yt_token = self.Parameters.yt_token_secret.data()['yt_token']

        bsdb_credentials = self.Parameters.bsdb_credentials_secret.data()
        bsdb_user = bsdb_credentials['user']
        bsdb_password = bsdb_credentials['password']

        circuit = self.Parameters.circuit
        yt_proxy = CIRCUIT_PARAMETERS[circuit]['yt_proxy']

        config_resource_type = CIRCUIT_PARAMETERS[circuit]['resource_type']
        LOGGER.info('Loading latest %r resource', config_resource_type)
        config_resource = sdk2.Resource[config_resource_type].find(
            attrs={'released': 'stable'},
        ).first()
        config_path = str(sdk2.ResourceData(config_resource).path)
        LOGGER.info('Resource %r loaded to path %r', config_resource.id, config_path)

        rpl_config = ReplicatorConfig.from_json(config_path)
        monitoring = ReplicaMonitoring(
            rpl_config,
            yt_proxy,
            yt_token,
            bsdb_user,
            bsdb_password,
        )
        row_counts = monitoring.get_row_counts()
        row_count_diffs = monitoring.calc_row_count_diffs(row_counts)

        if self.Parameters.should_push_to_solomon:
            solomon_token = self.Parameters.solomon_token_secret.data()['solomon_token']
            current_time = datetime.utcnow()
            solomon_client = SolomonClient(
                circuit=circuit,
                is_production=self.Parameters.solomon_instance == 'prod',
                token=solomon_token,
            )
            solomon_client.push_row_counts(row_counts, current_time)
            solomon_client.push_row_count_diffs(row_count_diffs, current_time)
