# -*- coding: utf-8 -*-
import logging
from sandbox import sdk2
import time
from sandbox.sandboxsdk import environments


class YabsPrepareDynamicFeed(sdk2.Task):
    """
        https://st.yandex-team.ru/BSDEV-68476
    """

    class Parameters(sdk2.Task.Parameters):
        cluster = sdk2.parameters.String("Cluster to process", required=True)
        cluster_to_write = sdk2.parameters.String("Cluster to write", required=True)
        yql_token = sdk2.parameters.String("YQL Token vault name", required=True)
        yt_token = sdk2.parameters.String("YT Token vault name", required=True)

    class Requirements(sdk2.Task.Requirements):
        environments = (
            environments.PipEnvironment('yandex-yt', '0.8.38a1', use_wheel=True),
            environments.PipEnvironment('yandex-yt-yson-bindings-skynet', use_wheel=True),
            environments.PipEnvironment('yql', version='1.2.91', use_wheel=True),
        )

        cores = 1
        ram = 4096
        disk_space = 4096

        class Caches(sdk2.Requirements.Caches):
            pass

    def _create_yt_client(self):
        import yt.wrapper as yt

        token = sdk2.Vault.data(self.Parameters.yt_token)
        return yt.YtClient(config={"token": token, "proxy": {"url": self.Parameters.cluster_to_write}})

    def _create_yql_client(self):
        from yql.api.v1.client import YqlClient

        token = sdk2.Vault.data(self.Parameters.yql_token)
        return YqlClient(db=self.Parameters.cluster, token=token)

    def _push(self, yt_client, table, rows):
        Complete = False
        try_count = 0
        logging.info("push rows " + table + " " + str(len(rows)))
        while not Complete:
            try:
                yt_client.insert_rows(table, rows, update=True)
                Complete = True
            except Exception:
                try_count = try_count + 1
                logging.warning("falied count {0}".format(try_count))
                time.sleep(10)

    def on_execute(self):
        table_path = "//home/yabs/dict/DynamicTemplateFeed"

        yt_client = self._create_yt_client()

        schema = [
            {'expression': 'farm_hash(BannerID)', 'sort_order': 'ascending', 'type': 'uint64', 'name': 'YTHash', 'required': False},
            {'name': 'BannerID', 'type': 'int64',  'sort_order': 'ascending', 'required': False},
            {'name': 'IsFromFeed', 'type': 'boolean', 'required': False},
        ]

        yt_client.create(path=table_path, type="table", attributes={"schema": schema, "dynamic": True}, ignore_existing=True)

        yt_client.mount_table(path=table_path)

        yql = self._create_yql_client()

        request = yql.query("""
            PRAGMA File('bigb.so', 'https://proxy.sandbox.yandex-team.ru/last/BIGB_UDF?attrs={"released":"stable"}&cache=1235435');
            PRAGMA udf('bigb.so');

            $caesar = '//home/bs/logs/AdsCaesarBannersFullDump/latest';

            $banners =
            SELECT
                BannerID
            FROM
                `//home/yabs-cs/export/YTBanner`
            WHERE
                OptionsDynamicTemplate
            AND
                NOT OptionsPerformance
            AND
                ParentExportID > 0
            GROUP BY
                BannerID
            ;

            $caesar_banners =
            SELECT
                BannerID,
                IF (Bigb::ParseBannerProfile(TableRow()).Resources.DirectBannersLogFields.BannerLandData LIKE "%FeedUrl%", true, false) AS IsFromFeed
            FROM
                $caesar
            WHERE
                Bigb::ParseBannerProfile(TableRow()).Resources.DirectBannersLogFields.BannerLandData IS NOT NULL;
            ;

            SELECT
                t1.BannerID AS BannerID,
                IsFromFeed
            FROM
                $caesar_banners AS t1
            JOIN
                $banners AS t2
            ON
                t1.BannerID == t2.BannerID
            ;
        """, syntax_version=1)

        result = request.run()

        tables_to_delete = result.get_results()

        if not tables_to_delete.is_success:
            raise RuntimeError("Query execution failed")

        yt_rows = list()

        for table in tables_to_delete:
            for row in table.get_iterator():
                yt_rows.append({"BannerID": int(row[0]), "IsFromFeed": bool(row[1])})

                if len(yt_rows) > 10000:
                    self._push(yt_client, table_path, yt_rows)
                    yt_rows = list()

        self._push(yt_client, table_path, yt_rows)
