import logging
from datetime import datetime

from sandbox import sdk2
from sandbox.projects.yabs.sbyt.lib.solomon_client import YabsStatSolomonClientBase
from sandbox.sandboxsdk import environments

LOGGER = logging.getLogger(__name__)


class YabsQTailStatMonitoring(sdk2.Task):
    """
        BannerQTailStatInterval data monitoring
    """

    class Parameters(sdk2.Task.Parameters):
        description = 'BannerQTailStatInterval data monitoring'

        yql_token_secret = sdk2.parameters.YavSecret(
            'Yav secret with YQL token',
            required=True,
        )

        solomon_token_secret = sdk2.parameters.YavSecret(
            'Yav secret with Solomon token',
            required=True,
        )

        with sdk2.parameters.RadioGroup('Solomon instance') as solomon_instance:
            solomon_instance.values['preprod'] = solomon_instance.Value('Prestable', default=True)
            solomon_instance.values['prod'] = solomon_instance.Value('Production')

    class Requirements(sdk2.Task.Requirements):
        environments = (
            environments.PipEnvironment('yandex-yt', version='0.8.38a1'),
            environments.PipEnvironment('yandex-yt-yson-bindings-skynet'),
            environments.PipEnvironment('yql', version='1.2.91'),
            environments.PipEnvironment('solomon')
        )

    class QTailSolomonClient(YabsStatSolomonClientBase):
        DEF_LABELS = {
            'metric': 'BannerQTailStatControl_Sum'
        }
        QTAIL_TYPE = {
            1: 'QTailID = 1',
            0: 'QTailID = 0',
            -1: 'QTailID > 1',
        }

        def __init__(self, cluster, is_production=False, token=None):
            YabsStatSolomonClientBase.__init__(self, "qtail", cluster, is_production, token)

    def _create_yql_client(self):
        from yql.api.v1.client import YqlClient

        token = self.Parameters.yql_token_secret.data()['yql_token']
        return YqlClient(token=token)

    def _send_metrics(self, cluster, stat):
        solomon_token = self.Parameters.solomon_token_secret.data()['solomon_token']
        s_cli = YabsQTailStatMonitoring.QTailSolomonClient(
            cluster=cluster,
            is_production=self.Parameters.solomon_instance == 'prod',
            token=solomon_token,
        )
        for row in stat:
            labels = dict(YabsQTailStatMonitoring.QTailSolomonClient.DEF_LABELS)
            labels['QTailType'] = YabsQTailStatMonitoring.QTailSolomonClient.QTAIL_TYPE[int(row['QTailType'])]
            sensors = [
                'DShows',
                'DClicks',
                'PShows',
                'PClicks',
            ]
            values = []
            for s in sensors:
                values.append(int(row[s]))

            s_cli.push_metrics(
                sensors,
                values,
                [labels] * len(sensors),
                datetime.utcfromtimestamp(int(row['UpdateTime']))
            )

    YQL_OPERATION_PATTERN = 'https://yql.yandex-team.ru/Operations/{id}'

    def _get_share_url(self, operation):
        from yql.client.operation import YqlOperationShareIdRequest

        share_request = YqlOperationShareIdRequest(operation.operation_id)
        share_request.run()

        return self.YQL_OPERATION_PATTERN.format(id=share_request.json)

    def _yql_operation_callback(self, operation):
        self.operation_id = operation.operation_id
        share_url = self._get_share_url(operation)
        LOGGER.info('YQL shared link : %s', share_url)

    def on_execute(self):
        yql = self._create_yql_client()
        clusters = [
            'hahn',
            'arnold',
            'freud'
        ]

        query_text = """
$date_from = DateTime::ToSeconds(DateTime::MakeTzDatetime(DateTime::StartOfDay(AddTimezone(CurrentUtcTimestamp(),"Europe/Moscow")))) - 3 * 60 * 60 * 24;
"""

        for cluster in clusters:
            query_text = query_text + """
SELECT
    SUM(PShows) as PShows,
    SUM(PClicks) as PClicks,
    SUM(DShows) as DShows,
    SUM(DClicks) as DClicks,
    QTailType,
    CTRDiff,
    UpdateTime
FROM
    {}.`//home/yabs/stat/BannerQTailStatInterval`
WHERE
    UpdateTime >= $date_from
GROUP BY
    UpdateTime,
    IF(QTailID > 1, -1, CAST(QTailID as Int64)) as QTailType,
    (CTRDiff > 0) AS CTRDiff
;
""".format(cluster)

        query = yql.query(query_text, syntax_version=1)
        query.run(pre_start_callback=self._yql_operation_callback)
        results = query.get_results()

        if not results.is_success:
            msg = '\n'.join([str(err) for err in results.errors])
            LOGGER.error('Error when executing query: %s', msg)
            raise Exception(msg)
        else:
            LOGGER.info('Query successfully finished. Collecting result')

        for (cluster, table) in zip(clusters, results):
            result = []
            table.fetch_full_data()
            columns = []
            for column_name, column_type in table.columns:
                columns.append(column_name)
            for row in table.rows:
                result.append(dict([(columns[i], value) for i, value in enumerate(row)]))

            self._send_metrics(cluster, result)
