"""Yabs SBYT runner."""
import subprocess

from sandbox.common import errors as ce
from sandbox.common.types import client as ctc
from sandbox.sandboxsdk import environments
from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk import task

from sandbox.projects.common import utils
from sandbox.projects.common.yabs import app_options
from sandbox.projects.common.yabs import ytstat


LOG_FILE = 'runner.log'


class CollectorGroup(parameters.SandboxStringParameter):
    name = 'collector_group'
    description = 'Collector Group'
    group = 'Collector'
    required = True


class CollectorName(parameters.SandboxStringParameter):
    name = 'collector_name'
    description = 'Collector Name'
    group = 'Collector'
    required = True


class YtProxy(parameters.SandboxStringParameter):
    name = 'yt_proxy'
    description = 'YT proxy'
    group = 'Collector'
    required = True


class YtTokenVaultName(parameters.SandboxStringParameter):
    name = 'yt_token_vault_name'
    description = 'YT token vault name'
    group = 'Collector'


class InputPattern(parameters.SandboxStringParameter):
    name = 'input_pattern'
    description = 'Input regexp'
    group = 'Collector'
    required = True


class InvocationID(parameters.SandboxIntegerParameter):
    name = 'invocation_id'
    description = 'InvocationID'
    group = 'Collector'


class Stage(parameters.SandboxStringParameter):
    name = 'stage'
    description = 'Stage'
    group = 'Collector'


class DestPath(parameters.SandboxStringParameter):
    name = 'yt_dest_path'
    description = 'YT destination path'
    group = 'Collector'


class DictPath(parameters.SandboxStringParameter):
    name = 'yt_dict_path'
    description = 'YT dict path'
    group = 'Collector'


class TmpPath(parameters.SandboxStringParameter):
    name = 'yt_tmp_path'
    description = 'YT tmp path'
    group = 'Collector'


class RplPath(parameters.SandboxStringParameter):
    name = 'yt_rpl_path'
    description = 'YT replica path'
    group = 'Collector'


class Options(parameters.DictRepeater, parameters.SandboxStringParameter):
    name = 'options'
    description = 'Custom options'
    group = 'Collector'


class NeedSchematize(parameters.SandboxBoolParameter):
    name = 'need_schematize'
    description = 'Should we schematize from logfeller format?'
    default_value = False
    group = 'Collector'


class CollectorReleaseYtProxy(parameters.SandboxStringParameter):
    name = 'collector_release_yt_proxy'
    description = 'Yt Proxy that has collector release (default is Yt Proxy)'
    group = 'Settings'


class CollectorResourceID(parameters.SandboxIntegerParameter):
    name = 'collector_resource_id'
    description = 'Collector Resource ID'
    group = 'Settings'


class LogsChunkSize(parameters.SandboxIntegerParameter):
    name = 'logs_chunk_size'
    description = 'How many logs will be processed at once (0 -- for all)'
    default_value = 1
    group = 'Settings'


class ShouldRemoveTmpPath(parameters.SandboxBoolParameter):
    name = 'should_remove_tmp_path'
    description = 'Should we remove tmp path?'
    default_value = True
    group = 'Settings'


class YabsSbytRunner(task.SandboxTask):
    """Task gives possibility to run SBYT's collector without Coordinator."""

    client_tags = ctc.Tag.LINUX_PRECISE
    cores = 1

    environment = (
        environments.PipEnvironment('yandex-yt', '0.8.38a1', use_wheel=True),
        environments.PipEnvironment('yandex-yt-yson-bindings-skynet', use_wheel=True),
    )
    input_parameters = (
        CollectorGroup,
        CollectorName,
        YtProxy,
        YtTokenVaultName,
        InputPattern,
        InvocationID,
        Stage,
        DestPath,
        DictPath,
        TmpPath,
        RplPath,
        Options,
        NeedSchematize,
        CollectorReleaseYtProxy,
        CollectorResourceID,
        LogsChunkSize,
        ShouldRemoveTmpPath,
    )
    type = 'YABS_SBYT_RUNNER'

    def initCtx(self):
        self.ctx['kill_timeout'] = 24 * 60 * 60  # 1 day

    def on_execute(self):
        ytstat.prepare_log_file(self, LOG_FILE)
        with ytstat.stderr_redirect(LOG_FILE):
            self._do_on_execute()

    def _do_on_execute(self):
        collector_group = utils.get_or_default(self.ctx, CollectorGroup)
        collector_name = utils.get_or_default(self.ctx, CollectorName)
        yt_proxy = utils.get_or_default(self.ctx, YtProxy)
        yt_token_vault_name = utils.get_or_default(self.ctx, YtTokenVaultName)
        input_pattern = utils.get_or_default(self.ctx, InputPattern)
        options = utils.get_or_default(self.ctx, Options)
        invocation_id = utils.get_or_default(self.ctx, InvocationID)
        stage = utils.get_or_default(self.ctx, Stage)
        dest_path = utils.get_or_default(self.ctx, DestPath)
        dict_path = utils.get_or_default(self.ctx, DictPath)
        tmp_path = utils.get_or_default(self.ctx, TmpPath)
        rpl_path = utils.get_or_default(self.ctx, RplPath)
        need_schematize = utils.get_or_default(self.ctx, NeedSchematize)

        collector_release_yt_proxy = utils.get_or_default(self.ctx, CollectorReleaseYtProxy)
        collector_resource_id = utils.get_or_default(self.ctx, CollectorResourceID)
        logs_chunk_size = utils.get_or_default(self.ctx, LogsChunkSize)
        should_remove_tmp_path = utils.get_or_default(self.ctx, ShouldRemoveTmpPath)

        yt_token = self.get_vault_data('YABS-YTSTAT', yt_token_vault_name) if yt_token_vault_name else None

        ytc = ytstat.get_yt_client(yt_proxy, yt_token)
        ytstat.install_supervisor(self, ytc)

        if not collector_resource_id:
            from supervisor import consts

            release_yt_proxy = collector_release_yt_proxy or ytc
            collector_resources = ytstat.get_autoreleased_resources(consts.COLLECTOR_RESOURCE_TYPE, release_yt_proxy)
            collector_resource = next((r for r in collector_resources if r.attributes['subtype'] == collector_group))
            collector_resource_id = collector_resource.id

        ytstat.install_resource(self, collector_resource_id)

        app_options.AppOptions.get_app_options_resource(self)
        app_options_file = app_options.AppOptions.make_app_options_file(yt_proxy)
        env = app_options.AppOptions.get_env_variables(app_options_file)

        from supervisor import ylogging

        logger = ylogging.get_logger()

        # TODO: see note in the head of files
        # from supervisor import runner
        from . import _runner as runner

        if yt_token:
            env['YT_TOKEN'] = yt_token

        try:
            runner.run(
                self,
                ytc,
                collector_group, collector_name,
                yt_proxy, input_pattern,
                app_options_file, options,
                invocation_id, stage,
                dest_path, dict_path, tmp_path, rpl_path, need_schematize,
                env,
                logs_chunk_size, should_remove_tmp_path,
            )
        except subprocess.CalledProcessError as e:
            logger.exception('collector error:')
            raise ce.TemporaryError(str(e))


__Task__ = YabsSbytRunner
