import time
import logging
import textwrap as tw
from pprint import pformat
from sandbox import sdk2
from sandbox.sdk2 import parameters
from sandbox.projects.yabs.base_bin_task import BaseBinTask


AUTOBUDGET_EXTERNAL_STAT_QUERY = '''
    $now = DateTime::ToSeconds(CurrentUtcTimestamp());

    SELECT
        `OrderID`, `StartTime`, `StatName`, `IntervalStartTime`
    FROM
        `{path}`
    WHERE
        (IntervalStartTime > 0 and IntervalStartTime < {action_min_time}
            and StatName in (\"actionweek\", \"actionday\"))
        or (IntervalStartTime > 0 and IntervalStartTime < {versioned_filter_time}
            and StatName in (\"day\", \"week\", \"dayweek\", \"group\", \"groupweek\"))'''


PREBILLING_STAT_QUERY = '''
    $now = DateTime::ToSeconds(CurrentUtcTimestamp());

    SELECT
        `SuperKeyHash`, `OrderID`, `Hash`
    FROM
        `{path}`
    WHERE
        (KeyFormat=\"ActionStatusValues:GroupOrderID,AttributionType,GoalID,VisitUserID,VisitID\"
            and Yson::ConvertToUint64(Yson::ConvertToDict(Value)[\"GoalEventTime\"]) < {action_min_time})
        or (KeyFormat=\"ActionStatusValuesV2:OrderID,LogID,AttributionType,GoalID,VisitUserID,VisitID\"
            and Yson::ConvertToUint64(Yson::ConvertToDict(Value)[\"GoalEventTime\"]) < {action_min_time})
        or (KeyFormat=\"AutoBudgetActionsWeeklyValues:OrderID,AttributionType,AutobudgetWeekBudgetCur,AutobudgetAvgCpaCur,AutobudgetStartTime,AutobudgetGoalID,WeekStartTime\"
            and Yson::ConvertToUint64(Yson::ConvertToDict(Value)[\"WeekStartTime\"]) < {action_min_time})
        or (KeyFormat=\"AutoBudgetStatVersionedCommon:OrderID,IntervalStartTime,StartTime,StatName\"
            and Yson::ConvertToUint64(Yson::ConvertToDict(Value)[\"IntervalStartTime\"]) < {versioned_filter_time}
            and Yson::ConvertToUint64(Yson::ConvertToDict(Value)[\"IntervalStartTime\"]) > 0
            and Yson::ConvertToString(Yson::ConvertToDict(Value)[\"StatName\"]) in (\"day\", \"week\", \"dayweek\", \"group\", \"groupweek\"))
        or (KeyFormat=\"AutoBudgetStatVersionedCommon:OrderID,IntervalStartTime,StartTime,StatName\"
            and Yson::ConvertToUint64(Yson::ConvertToDict(Value)[\"IntervalStartTime\"]) < {action_min_time}
            and Yson::ConvertToUint64(Yson::ConvertToDict(Value)[\"IntervalStartTime\"]) > 0
            and Yson::ConvertToString(Yson::ConvertToDict(Value)[\"StatName\"]) in (\"actionweek\", \"actionday\"))
        or (KeyFormat=\"AutoBudgetStrategyStatVersionedCommon:OrderID,IntervalStartTime,StartTime,AutoBudgetStrategyID,StatName\"
            and Yson::ConvertToUint64(Yson::ConvertToDict(Value)[\"IntervalStartTime\"]) < {versioned_filter_time}
            and Yson::ConvertToUint64(Yson::ConvertToDict(Value)[\"IntervalStartTime\"]) > 0
            and Yson::ConvertToString(Yson::ConvertToDict(Value)[\"StatName\"]) in (\"day\", \"week\"))
        or (KeyFormat=\"AutoBudgetStrategyStatVersionedCommon:OrderID,IntervalStartTime,StartTime,AutoBudgetStrategyID,StatName\"
            and Yson::ConvertToUint64(Yson::ConvertToDict(Value)[\"IntervalStartTime\"]) < {action_min_time}
            and Yson::ConvertToUint64(Yson::ConvertToDict(Value)[\"IntervalStartTime\"]) > 0
            and Yson::ConvertToString(Yson::ConvertToDict(Value)[\"StatName\"]) in (\"actionweek\"))
        '''


DAY = 86400
YEAR = 366 * DAY
PREDEFINED_CONFIG = {
    'search_query_md5': {
        'yql_query': tw.dedent('''\
            SELECT
                `SearchQueryMD5`
            FROM `//home/yabs/stat/SearchQueryByMD5`
            WHERE (`LastCheventTime` < {drop_time}) or (`LastCheventTime` is null and `UpdateTime` < {drop_time});
        '''),
        'table_path': '//home/yabs/stat/replica/SearchQueryByMD5',
        'delta_time': 3 * YEAR
    },
    'autobudget_stat_external_prestable': {
        'yql_query': tw.dedent(AUTOBUDGET_EXTERNAL_STAT_QUERY).format(
            path="//home/yabs/dict/Fury/PreprodAutoBudgetStatExternal",
            action_min_time=int(time.time()) - 180 * DAY,
            versioned_filter_time=int(time.time()) - 30 * DAY),
        'table_path': '//home/yabs/dict/Fury/PreprodAutoBudgetStatExternal'
    },
    'autobudget_stat_external': {
        'yql_query': tw.dedent(AUTOBUDGET_EXTERNAL_STAT_QUERY).format(
            path="//home/yabs/dict/Fury/AutoBudgetStatExternal",
            action_min_time=int(time.time()) - 180 * DAY,
            versioned_filter_time=int(time.time()) - 30 * DAY),
        'table_path': '//home/yabs/dict/Fury/AutoBudgetStatExternal'
    },
    'prebilling_stat_prestable': {
        'yql_query': tw.dedent(PREBILLING_STAT_QUERY).format(
            path="//home/antifraud/fury/rtprebilling/preprod/history/order_stat",
            action_min_time=int(time.time()) - 180 * DAY,
            versioned_filter_time=int(time.time()) - 30 * DAY),
        'table_path': '//home/antifraud/fury/rtprebilling/preprod/history/order_stat'
    },
    'prebilling_stat': {
        'yql_query': tw.dedent(PREBILLING_STAT_QUERY).format(
            path="//home/antifraud/fury/rtprebilling/prod/history/order_stat",
            action_min_time=int(time.time()) - 180 * DAY,
            versioned_filter_time=int(time.time()) - 30 * DAY),
        'table_path': '//home/antifraud/fury/rtprebilling/prod/history/order_stat'
    }
}
CLUSTERS = ['freud', 'zeno', 'hahn', 'arnold', 'senecasas', 'senecaman', 'senecavla', 'bohr', 'landau', 'markov']


class YabsYtStatTableCleaner(BaseBinTask):
    class Requirements(sdk2.Requirements):
        cores = 1
        ram = 4096
        disk_space = 4096

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(BaseBinTask.Parameters):
        description = 'table cleaner task'

        with BaseBinTask.Parameters.version_and_task_resource() as version_and_task_resource:
            resource_attrs = parameters.Dict('Filter resource by', default={'name': 'YabsYtStatTableCleaner'})

        yt_token_secret = parameters.YavSecret(
            'YT  token secret',
            default='sec-01d4mdr98tm9n1k6a3rmyn685t#YT_TOKEN',
            required=True
        )

        yql_token_secret = parameters.YavSecret(
            'YQL token secret',
            default='sec-01d4mdr98tm9n1k6a3rmyn685t#YQL_TOKEN',
            required=True
        )

        with parameters.String('search table type to clean', multiline=True, required=True) as predefined_config:
            for key, val in PREDEFINED_CONFIG.items():
                predefined_config.values[key] = pformat(val)

        with parameters.String('cluster with table for delete rows', multiline=True, required=True) as yt_cluster:
            for cluster in CLUSTERS:
                yt_cluster.values[cluster] = cluster

        with parameters.String('cluster with table for yql select', multiline=True, required=True) as yql_select_cluster:
            for cluster in CLUSTERS:
                yql_select_cluster.values[cluster] = cluster

        num_rows_in_one_delete_rows = parameters.Integer(
            'num rows in one delete rows',
            default=100000,
        )

        sleep_time = parameters.Integer(
            'sleep time in seconds between delete rows',
            default=1,
        )

        max_delete_rows_runs_per_task_run = parameters.Integer(
            'max delete rows per one run (0 = inf)',
            default=20,
        )

    def on_execute(self):
        from yql.api.v1.client import YqlClient
        from yt.wrapper import YtClient

        query_and_table_as_dict = PREDEFINED_CONFIG[self.Parameters.predefined_config]
        table_path = query_and_table_as_dict['table_path']

        if 'delta_time' in query_and_table_as_dict:
            drop_time = time.time() - query_and_table_as_dict['delta_time']
            yql_query = query_and_table_as_dict['yql_query'].format(drop_time=drop_time)
        else:
            yql_query = query_and_table_as_dict['yql_query']

        self.yt_token = self.Parameters.yt_token_secret.data()[self.Parameters.yt_token_secret.default_key]
        self.yql_token = self.Parameters.yql_token_secret.data()[self.Parameters.yql_token_secret.default_key]
        yql_client = YqlClient(db=self.Parameters.yql_select_cluster, token=self.yql_token)
        request = yql_client.query(yql_query, syntax_version=1)
        yt_client = YtClient(token=self.yt_token, proxy=self.Parameters.yt_cluster)
        logging.info("Start running YQL")
        results = request.run().get_results()
        logging.info("YQL finished")
        rows = []
        delete_rows_count = 0
        for table in results:
            logging.info("Start process table")
            for row in table.get_iterator():
                rows.append(dict(zip(table.column_names, row)))
                if len(rows) >= self.Parameters.num_rows_in_one_delete_rows:
                    logging.info("Collected {} rows iter {}".format(len(rows), delete_rows_count))
                    delete_rows_count += 1
                    yt_client.delete_rows(table_path, rows, require_sync_replica=False)
                    logging.info("Collected rows were deleted")
                    rows = []
                    if (self.Parameters.max_delete_rows_runs_per_task_run > 0) and \
                         (delete_rows_count > self.Parameters.max_delete_rows_runs_per_task_run - 1):
                        logging.info("Max delete count rows were made")
                        break
                    time.sleep(self.Parameters.sleep_time)
                    logging.info("Process woke up after sleep")

        if len(rows) > 0:
            logging.info("Delete rest {} rows".format(len(rows)))
            yt_client.delete_rows(table_path, rows, require_sync_replica=False)
