import os
import os.path
import re
import shutil
import logging

from sandbox import sdk2
import sandbox.common.types.client as ctc
import sandbox.common.types.misc as ctm
import sandbox.common.types.notification as ctn
from sandbox.sdk2.helpers import subprocess as sp, gdb, ProcessLog


logger = logging.getLogger('sandbox-task')
logger.setLevel(logging.DEBUG)


logger_sp = logging.getLogger('sandbox-subprocess')
logger_sp.setLevel(logging.DEBUG)


PRIVATE_KEY_FILE_PATH = "./private_key"
CERTIFICATES_DIRECTORY = "./roles/nginx/files/temporary"
SECRET_ID = "sec-01e6hywv1h3x29ktzzk2bsst79"


class YabsServerSspAnsible(sdk2.Task):
    class Requirements(sdk2.Requirements):
        container_resource = 2396959152
        cores = 1
        disk_space = 1024
        dns = ctm.DnsType.DNS64  # need NAT64
        client_tags = ctc.Tag.LINUX_XENIAL  # for ansible package

        class Caches(sdk2.Requirements.Caches):
            pass  # means that task do not use any shared caches

    class Parameters(sdk2.Parameters):
        playbook_name = sdk2.parameters.String('Name of playbook to run', default_value='playbook_ssp_proxy.yml')
        arcadia_path = sdk2.parameters.String('Arcadia path', default_value="arcadia:/arc/trunk/arcadia/yabs/stat/infra/ssp_proxy/ansible")
        tags_filter = sdk2.parameters.String('Filter tasks via tags')
        hosts_filter = sdk2.parameters.String('Filter hosts')
        check = sdk2.parameters.Bool('Run in CHECK mode', default_value=True)
        diff = sdk2.parameters.Bool('Run in DIFF mode', default_value=True)
        release_initiator = sdk2.parameters.String('Release initiator - will be added to configuration changelog as "author"')
        notify_on_diff = sdk2.parameters.Bool('Notify on DIFF', default_value=False)

        with sdk2.parameters.Output():
            has_diff = sdk2.parameters.Bool("has_diff")
            hosts_filter_out = sdk2.parameters.String("hosts_filter_out")

    def on_execute(self):
        logger.debug('checkout arcadia')
        sdk2.svn.Arcadia.checkout(
            self.Parameters.arcadia_path,
            '.', sdk2.svn.Arcadia.Depth.INFINITY
        )

        logger.debug('install ssh private key')
        id_rsa = sdk2.yav.Secret(SECRET_ID).data()["id_rsa"]
        with open(PRIVATE_KEY_FILE_PATH, "w") as id_rsa_out:
            id_rsa_out.write(id_rsa)
        os.chmod(PRIVATE_KEY_FILE_PATH, 0o600)

        logger.debug('prepare certificate files')
        if "ssp_proxy" in self.Parameters.playbook_name:
            certificates_secret_id = "sec-01e6hywv1h3x29ktzzk2bsst79"
            certificates_keys = ["bs.yandex.ru.key", "bs.yandex.ru.pem"]
        elif "postback_proxy" in self.Parameters.playbook_name:
            certificates_secret_id = "sec-01f277r90xw0r3fd33kk4y2vhr"
            certificates_keys = ["postback.yandexadexchange.net.key", "postback.yandexadexchange.net.pem"]
        else:
            raise Exception("Can't find certificates secret for playbook!")
        certificates_secret = sdk2.yav.Secret(certificates_secret_id).data()
        os.makedirs(CERTIFICATES_DIRECTORY)
        for key in certificates_keys:
            with open("{}/{}".format(CERTIFICATES_DIRECTORY, key), "w") as key_out:
                key_out.write(certificates_secret[key])
                key_out.write("\n")

        logger.debug('set sandbox token')
        sandbox_token = sdk2.yav.Secret(SECRET_ID).data()["SANDBOX_TOKEN"]
        env = {}
        env.update(os.environ)
        env["SANDBOX_TOKEN"] = sandbox_token

        if self.Parameters.release_initiator:
            env["RELEASE_INITIATOR"] = self.Parameters.release_initiator

        rev_info = sdk2.svn.Arcadia.info('.')
        logger.debug('arcadia revision: %s', rev_info)
        msg_lines = [
            'ARCADIA REVISION:',
            'author: %s' % rev_info['author'],
            'commit revision: %s' % rev_info['commit_revision'],
            'link: https://a.yandex-team.ru/arc/commit/%s' % rev_info['commit_revision'],
        ]
        self.set_info('\n'.join(msg_lines))
        env["SVN_REVISION"] = rev_info['commit_revision']

        output = sp.check_output(['ansible', '--version']).strip()
        self.set_info('ANSIBLE VERSION:\n%s' % output)
        logger.debug('ansible version: %s', ', '.join(re.split(r'\s\s+', output)))

        env['ANSIBLE_CONFIG'] = os.path.join(os.path.abspath('.'), 'ansible.cfg')

        cmd = ['ansible-playbook']
        cmd.append("--private-key {}".format(PRIVATE_KEY_FILE_PATH))
        if self.Parameters.tags_filter:
            cmd.append('--tags {}'.format(self.Parameters.tags_filter))
        if self.Parameters.hosts_filter:
            cmd.append('--limit {}'.format(self.Parameters.hosts_filter))
        if self.Parameters.check:
            cmd.append("--check")
        if self.Parameters.diff:
            cmd.append("--diff")
        cmd.append(str(self.Parameters.playbook_name))

        self.set_info('EXECUTING:\n%s' % ' '.join(cmd))
        logger.info('executing: %s', cmd)

        try:
            process_log = ProcessLog(self, logger='ansible-playbook')
            self.set_info(
                gdb.get_html_view_for_logs_file(
                    "ansible_stdout",
                    process_log.stdout.path.relative_to(self.log_path()),
                    self.log_resource
                ), do_escape=False
            )
            with process_log:
                logger.debug('playbook execution starting')
                sp.check_call(' '.join(cmd), stdout=process_log.stdout, stderr=process_log.stderr, shell=True, env=env)
                logger.debug('playbook execution finished')

            has_diff = False
            self.Parameters.hosts_filter_out = self.Parameters.hosts_filter

            # check for diff
            with open(str(process_log.stdout.path)) as stdout_in:
                stdout_str = stdout_in.read()
            lines = [l for l in stdout_str.splitlines() if l.strip()]
            for line in reversed(lines):
                if "PLAY RECAP" in line:
                    break
                if "changed=0" not in line:
                    has_diff = True

            self.Parameters.has_diff = has_diff

        finally:
            os.remove(PRIVATE_KEY_FILE_PATH)
            if os.path.exists(CERTIFICATES_DIRECTORY):
                shutil.rmtree(CERTIFICATES_DIRECTORY)

        if self.Parameters.notify_on_diff and self.Parameters.has_diff:
            message = 'Found DIFF in ansible configuration!\nLink: https://sandbox.yandex-team.ru/task/{id}'.format(id=self.id)
            self.server.notification(
                body=message,
                recipients=["lazuka23", "vbryadov"],
                transport=ctn.Transport.TELEGRAM
            )
