import collections
import json
import logging
import requests
import time
from datetime import datetime, timedelta

from sandbox import sdk2
from sandbox.sandboxsdk import environments


class BaseYabsServerFillWhiteList(object):
    class Parameters(sdk2.Task.Parameters):

        with sdk2.parameters.Group('Common parameters') as common_params:
            sspid = sdk2.parameters.Integer('SSPID', required=True)
            white_ticket = sdk2.parameters.String('WhiteScript Task Ticket', required=True)
            priority_cluster = sdk2.parameters.String('Cluster to run yql queries', default='hahn', required=True)
            debug_mode = sdk2.parameters.Bool('Debug mode', default=False, required=True)

        with sdk2.parameters.Group('YQL parameters') as yql_params:
            yql_token_vault_name = sdk2.parameters.String('YQL Robot Token Vault Name', default='robot_google_yql_token', required=True)
            delta_days = sdk2.parameters.Integer('Delta logs dates', default=1, required=True)

        with sdk2.parameters.Group('Solomon parameters') as solomon_params:
            solomon_key = sdk2.parameters.YavSecret('Solomon Key', default='sec-01egr7r5qh453d0tq0q4bht99p', required=True)
            solomon_project = sdk2.parameters.String('Solomon Project ', default='yabs', required=True)
            solomon_cluster = sdk2.parameters.String('Solomon Cluster ', default='yabs_frontend_metadsp', required=True)
            solomon_service = sdk2.parameters.String('Solomon Service ', default='yabs_server', required=True)

        with sdk2.parameters.Group('YT Tables parameters') as table_params:
            wt_table_path = sdk2.parameters.String('White List Table Path ', default='//home/yabs/cs/SSPOrderWhiteList', required=True)

        with sdk2.parameters.Group('Order limits parameters') as order_limits_params:
            top_count = sdk2.parameters.Integer('TopCount of orders from chevent log', default=10000, required=True)

    class Requirements(sdk2.Task.Requirements):
        environments = (
            environments.PipEnvironment('yandex-yt'),
            environments.PipEnvironment('yql'),
        )
        cores = 4  # 4 cores or less
        ram = 8192  # 8GiB or less

        class Caches(sdk2.Requirements.Caches):
            pass

    class Context(sdk2.Task.Context):
        yt_clusters = ['hahn', 'arnold']
        cur_time = 0
        yql_ids = {}

    def get_process_time(self, fmt='%Y-%m-%d', days=1):
        start_time = datetime.now() - timedelta(days=days)
        return start_time.strftime(fmt)

    def get_solomon_data(self):
        raise NotImplementedError()

    def send_data_to_solomon(self):
        logging.info('Sending data to Solomon...')

        data = self.get_solomon_data()

        if self.Parameters.debug_mode:
            logging.info('Debug mode ON, skip sending stats')
            logging.info(data)
            return

        SOLOMON_URL = 'https://solomon.yandex.net/api/v2/push?project={project}&cluster={cluster}&service={service}'.format(
            project=self.Parameters.solomon_project,
            cluster=self.Parameters.solomon_cluster,
            service=self.Parameters.solomon_service
        )
        headers = {
            'Content-type': 'application/json',
            'Authorization': 'OAuth {token}'.format(token=self.Parameters.solomon_key.data()['token'])
        }

        response = requests.post(SOLOMON_URL, data=json.dumps(data), headers=headers)
        response.raise_for_status()

    def yt_connect(self, yt_cluster):
        import yt.wrapper as yt
        try:
            logging.info('Trying to connect to ' + str(yt_cluster))
            cfg = {
                'tabular_data_format': yt.JsonFormat(control_attributes_mode='row_fields'),
                'detached': False,
                'token': self.yql_token,
                'proxy': {'url': yt_cluster},
            }
            client = yt.YtClient(config=cfg)
            logging.info('Successfully connected to ' + str(yt_cluster))
            return client
        except:
            pass

    def run_yql_query(self, query, token, db, query_name):
        from yql.api.v1.client import YqlClient

        logging.info('Start yql query [{}]'.format(query_name))
        client = YqlClient(db=db, token=token)
        query = client.query(query, syntax_version=1)
        query.run()
        logging.info('Share YQL: {}'.format(query.share_url))
        self.Context.yql_ids[query_name] = query.operation_id

    def get_yql_results(self, query_name):
        from yql.client.operation import YqlOperationResultsRequest

        query = YqlOperationResultsRequest(self.Context.yql_ids[query_name])
        query.run()

        results = []
        for table in query.get_results():
            table.fetch_full_data()
            columns = []
            for column_name, column_type in table.columns:
                columns.append(column_name)
            for row in table.rows:
                results.append(dict([(columns[i], value) for i, value in enumerate(row)]))

        logging.info('Finish yql success [{}], collect {} rows'.format(query_name, len(results)))
        return results

    def get_chevent_log_table(self):
        with self.memoize_stage.get_chevent_table:
            self.Context.cur_time = int(time.time())
            start_process_time = self.get_process_time(days=self.Parameters.delta_days)
            self.Context.bs_chevent_log_table = '`logs/bs-chevent-log/1d/{start_process_time}`'.format(start_process_time=start_process_time)

    def write_white_list(self):
        logging.info('New orders {}'.format(self.new_orders))
        self.added_orders_by_cluster = collections.defaultdict(int)
        for yt_cluster in self.Context.yt_clusters:
            added_count = self.extend_white_list(yt_cluster, self.new_orders)
            if added_count is not None:
                logging.info('Successfully added {} orders to {}'.format(added_count, yt_cluster))
                self.added_orders_by_cluster[yt_cluster] = added_count
            else:
                logging.info('Failed to add orders to {}'.format(yt_cluster))

    def extend_white_list(self, yt_cluster, add_to_white_list):
        ytc = self.yt_connect(yt_cluster)
        rows = [{
            'SSPID': self.Parameters.sspid,
            'OrderID': order_id,
            'Ticket': self.Parameters.white_ticket,
            'Time': self.Context.cur_time
        } for order_id in add_to_white_list]

        if self.Parameters.debug_mode:
            logging.info('Debug mode ON, skip insert data')
            logging.info(rows)
            return

        if ytc and rows:
            ytc.insert_rows(self.Parameters.wt_table_path, rows)
            return len(rows)

    def run_subtasks(self):
        pass

    def run_queries(self):
        raise NotImplementedError()

    def get_data_from_yql(self):
        raise NotImplementedError()

    def fill_white_list(self):
        raise NotImplementedError()

    def on_execute(self):
        from yql.config import config

        logging.info('Started!')
        token_owner = getattr(self.Parameters, 'yql_token_owner', self.author)
        self.yql_token = sdk2.task.Vault.data(token_owner, self.Parameters.yql_token_vault_name)
        config.token = self.yql_token
        self.run_subtasks()
        self.get_chevent_log_table()
        self.run_queries()
        self.get_data_from_yql()
        self.fill_white_list()
        self.send_data_to_solomon()
        logging.info('Done!')
