#!/usr/bin/env python
# encoding=utf-8

from sandbox import sdk2
from sandbox.sandboxsdk import environments
import re
import logging
import json
import time

import orders


class YabsServerDeleteWhiteListLarge(sdk2.Task):

    YT_TABLE_SSP_WHITELIST = '//home/yabs/cs/SSPOrderWhiteList'

    class Parameters(sdk2.Task.Parameters):
        yql_token_vault_name = sdk2.parameters.String("YQL Robot Token Vault Name", default="robot_google_yql_token", required=True)
        yt_clusters = sdk2.parameters.String("YT clusters for read and apply (comma separated)", default="hahn,arnold", required=True)
        sspid = sdk2.parameters.Integer("SSPID", default=17298340, required=True)

        with sdk2.parameters.RadioGroup('Action', required=False) as action:

            action.values['add_to_whitelist'] = action.Value(value='add to whitelist')
            action.values['delete_from_whitelist'] = action.Value(value='delete from whitelist', default=False)

            with action.value['add_to_whitelist']:
                add_white_order_list = orders.orders

            with action.value['delete_from_whitelist']:
                delete_white_order_list = orders.orders

    class Requirements(sdk2.Task.Requirements):
        environments = (
            environments.PipEnvironment('yandex-yt'),
            environments.PipEnvironment('yql', version='1.2.91')
        )

    def yt_connect(self, yt_cluster):
        import yt.wrapper as yt

        logging.info("Try connect to " + str(yt_cluster))
        try:
            cfg = {
                "tabular_data_format": yt.JsonFormat(control_attributes_mode="row_fields"),
                "detached": False,
                "token": self.yql_token,
                "proxy": {"url": yt_cluster},

            }
            ytc = yt.YtClient(config=cfg)
            logging.info("Succesfully connected to {}".format(yt_cluster))

            return ytc
        except Exception as ex:
            logging.info("Can`t connect to {}".format(yt_cluster))
            logging.info(ex)
            return None

    def exec_yql_query(self, db, token, query):
        from yql.api.v1.client import YqlClient

        query = re.sub(r'\\', '', query)

        client = YqlClient(db=db, token=token)
        query = client.query(query, syntax_version=1)
        query.run()
        query.wait_progress()

        if not query.is_success:
            errors = []
            if query.errors:
                errors = [str(error) for error in query.errors]
            raise Exception("YQL request failed. Status: {}. Errors: {}".format(query.status, ' - '.join(errors)))

        result = []
        for table in query.get_results():
            table.fetch_full_data()

            columns = []
            for column_name, column_type in table.columns:
                columns.append(column_name)

            for row in table.rows:
                result.append(dict([(columns[i], value) for i, value in enumerate(row)]))

        return json.loads(json.dumps(result))

    def read_yt_table(self, yt_cluster, query):
        logging.info("Start reading yt table")

        yt_res = self.exec_yql_query(yt_cluster, self.yql_token, query)
        logging.info("OUT: {}\n".format(json.dumps(yt_res)))

        return yt_res

    def on_execute(self):
        self.yql_token = sdk2.task.Vault.data(self.author, self.Parameters.yql_token_vault_name)

        logging.info("Started action: " + str(self.Parameters.action))

        if self.Parameters.action == 'add_to_whitelist':
            self.action_add_to_whitelist()

        elif self.Parameters.action == 'delete_from_whitelist':
            self.action_delete_from_whitelist()

        logging.info("Done!")

    def action_delete_from_whitelist(self):
        sspid = self.Parameters.sspid
        candidates_list = self.Parameters.delete_white_order_list

        if len(candidates_list) > 0 and sspid > 0:
            for yt_cluster in self.Parameters.yt_clusters.split(","):
                logging.debug("Start action for yt-cluster: ".format(yt_cluster))
                cur_list = self.get_whitelist(yt_cluster, sspid, candidates_list)

                if cur_list:
                    for i in range(len(candidates_list) // 20000 + 1):
                        action_list = set(candidates_list[i * 20000:(i + 1) * 20000])
                        self.delete_from_whitelist(yt_cluster, sspid, action_list)

    def action_add_to_whitelist(self):
        sspid = self.Parameters.sspid
        candidates_list = self.Parameters.add_white_order_list

        if len(candidates_list) > 0 and sspid > 0:
            for yt_cluster in self.Parameters.yt_clusters.split(","):
                logging.debug("Start action for yt-cluster: ".format(yt_cluster))
                cur_list = self.get_whitelist(yt_cluster, sspid)

                if not len(cur_list):
                    logging.debug("YOU CANNOT ADD ORDERS TO EMPTY WHITELIST ( IT WILL TURN ON FILTERING )")
                    return None

                for i in range(len(candidates_list) // 20000 + 1):
                    action_list = set(candidates_list[i * 20000:(i + 1) * 20000]) - set(cur_list)
                    self.add_to_whitelist(yt_cluster, sspid, action_list)

    def get_whitelist(self, yt_cluster, sspid, orders_list):
        list = []
        ytc = self.yt_connect(yt_cluster)

        if sspid > 0 and ytc:
            where_cond = ''
            if len(orders_list) > 0:
                where_cond = " AND OrderID IN (" + ",".join([str(x) for x in orders_list]) + ")"

            list += [int(row['OrderID']) for row in ytc.select_rows("* from [{0}] where SSPID = {1} {2}".format(self.YT_TABLE_SSP_WHITELIST, sspid, where_cond))]
            return list
        else:
            return None

    def add_to_whitelist(self, yt_cluster, sspid, orders_list):
        ytc = self.yt_connect(yt_cluster)
        current_time = int(time.time())

        rows = [{'SSPID': int(sspid), 'OrderID': int(order_id), 'Ticket': 'SB_HELPER', 'Time': current_time} for order_id in orders_list]

        logging.debug({"candidates": orders_list, "rows": rows})

        if len(rows) > 0 and sspid > 0 and ytc and rows:
            logging.debug("Start insert operation")
            res = ytc.insert_rows(self.YT_TABLE_SSP_WHITELIST, rows)
            logging.debug("Inserted rows: {}".format(res))
            return res
        else:
            logging.info("Empty list")
            return None

    def delete_from_whitelist(self, yt_cluster, sspid, candidates_list):
        rows = []
        ytc = self.yt_connect(yt_cluster)

        for orid in candidates_list:
            rows += [{'SSPID': int(sspid), 'OrderID': int(orid)}]

        logging.debug({"candidates": candidates_list, "rows": rows})

        if ytc and rows:
            logging.debug("Start delete operation")
            res = ytc.delete_rows(self.YT_TABLE_SSP_WHITELIST, rows)
            logging.debug("Deleted rows: {}".format(res))
            return res
        else:
            logging.info("Empty rows")
            return None
