from sandbox import sdk2
from sandbox.sandboxsdk import environments
from sandbox.sandboxsdk.process import run_process

import logging
import time
import re
import json
import requests

from .yql_requests import GetFeedbackStatistics, GetBlackList, GetAllMalwareOrderByDomain, GetDomainsRequest
from .additional_functions import SendDataToSolomon, MakeNewBlackListRows, ParseBlackList, GetProcessTime, ParseDomains, MakeNewDomainRows

BATCH_SIZE = 50000

# Creative and Snippet Status Codes
# https://developers.google.com/authorized-buyers/rtb/response-guide/creative-snippet-codes
MALWARE_CODE = 6
STRICT_BAD_FEEDBACK_CODES = [
    MALWARE_CODE,  # Creative filtered because malware was detected
    10,  # Creative was not approved
]

RUSSIA_MODERATION_CODE = 130
FIRST_WORK_MODE = 1
SECOND_WORK_MODE = 2


class YabsServerFillBlackListNew(sdk2.Task):

    class Parameters(sdk2.Task.Parameters):
        solomon_key = sdk2.parameters.YavSecret("Solomon Key", default="sec-01egr7r5qh453d0tq0q4bht99p", required=True)
        yql_token_owner = sdk2.parameters.String("YQL Token Owner Name", default="robot-yabs-google", required=True)
        yql_token_vault_name = sdk2.parameters.String("YQL Robot Token Vault Name", default="robot_google_yql_token", required=True)
        google_sspid = sdk2.parameters.Integer("Google SSPID ", default=17298340, required=True)
        time_from_delay = sdk2.parameters.Integer("From time get feedback statistic ", default=240, required=True)
        cluster = sdk2.parameters.String("Working Cluster", default='hahn', required=True)
        black_list_table_path = sdk2.parameters.String("Black Table Path ", default="//home/yabs/cs/SSPOrderBlackList", required=True)
        domain_black_list_path = sdk2.parameters.String("Domain Black List Path ", default="//home/yabs-cs/data/SSPDomainBlackList", required=True)
        solomon_project = sdk2.parameters.String("Solomon Project ", default="yabs", required=True)
        solomon_cluster = sdk2.parameters.String("Solomon Cluster ", default="yabs_frontend_metadsp", required=True)
        solomon_service = sdk2.parameters.String("Solomon Service ", default="yabs_server", required=True)
        creatives_hist_tb_path = sdk2.parameters.String("History Table Path ", default="//home/yabs/cs/GoogleCreativesStatusHistoryNew", required=True)
        history_cluster = sdk2.parameters.String("History Table Cluster ", default="hahn", required=True)
        test_task = sdk2.parameters.Bool("test task ", default=True, required=True)
        min_bid_count = sdk2.parameters.Integer("Minimum bids for ban", default=5000, required=True)
        bad_moderation_rate = sdk2.parameters.Float("Minimum rate of unmoderated bids for ban", default=0.9, required=True)
        work_mode = sdk2.parameters.Integer("Work Mode, 1 - for delete moderation order in first account, 2 - delete malware from second account", default=1, required=True)
        good_domains = sdk2.parameters.String("Don't add these domains to black list (comma separated)", default="17242,4722329,10949129", required=True)

    class Requirements(sdk2.Task.Requirements):
        environments = (
            environments.PipEnvironment('yandex-yt'),
            environments.PipEnvironment('yql', version='1.2.91'),
        )

    def yt_connect(self, yt_cluster):
        import yt.wrapper as yt

        logging.info("Try connect to " + str(yt_cluster))
        try:
            cfg = {
                "tabular_data_format": yt.JsonFormat(control_attributes_mode="row_fields"),
                "detached": False,
                "token": self.yql_token,
                "proxy": {"url": yt_cluster},
            }
            ytc = yt.YtClient(config=cfg)
            logging.info("Succesfully connected to {}".format(yt_cluster))

            return ytc
        except:
            logging.info("Can`t connect to {}".format(yt_cluster))
            return None

    def run_yql_query(self, query):
        from yql.api.v1.client import YqlClient
        logging.info("start yql query")
        client = YqlClient(db=self.Parameters.cluster, token=self.yql_token)
        query = client.query(query, syntax_version=1)
        query.run()
        query.wait_progress()

        if not query.is_success:
            logging.info('\n'.join([str(err) for err in query.errors]))
            raise Exception()

        logging.info('parsing yql result')
        result = []
        for table in query.get_results():
            table.fetch_full_data()
            for row in table.rows:
                result.append(row)
        logging.info("finish yql success, collect {} rows".format(len(result)))
        return result

    def add_history_rows(self, creative_id, order_id, mistake, status):
        creatives = set([creative_id]) if creative_id != '' else self.all_orders_stats[order_id]['creatives']
        for creative_id in creatives:
            self.new_history_rows.append({
                'CreativeID': str(creative_id),
                'OrderID': str(order_id),
                'Time': str(self.current_time),
                'Mistake': str(mistake),
                'Status': status,
            })

    def create_order_stat(self, order_id):
        if order_id in self.all_orders_stats.keys():
            return
        self.all_orders_stats[order_id] = {
            'creatives': set([]),
            'mistakes': dict([]),
            'all_bids': 0,
        }

    def clear_black_list_table(self):
        for cluster in ['hahn', 'arnold']:
            ytc = self.yt_connect(cluster)
            if ytc and len(self.orders_to_delete_from_blacklist):
                try:
                    if self.Parameters.test_task:
                        logging.info('Delete rows from black_list {}'.format(self.orders_to_delete_from_blacklist))
                    else:
                        ytc.delete_rows(self.Parameters.black_list_table_path, self.orders_to_delete_from_blacklist)
                    logging.info('Delete {} rows from black_list'.format(len(self.orders_to_delete_from_blacklist)))
                except:
                    logging.info('Problem with {}'.format(cluster))

    def fill_black_list_table(self):
        new_black_list_rows = MakeNewBlackListRows(self.strict_bad_orders, self.temporary_bad_orders, self.Parameters.google_sspid, self.unix_current_time)
        for cluster in ['hahn', 'arnold']:
            ytc = self.yt_connect(cluster)
            if ytc and len(new_black_list_rows):
                if self.Parameters.test_task:
                    logging.info('Insert rows from black_list {}'.format(new_black_list_rows))
                else:
                    ytc.insert_rows(self.Parameters.black_list_table_path, new_black_list_rows)
                logging.info('Inserted {} rows'.format(len(new_black_list_rows)))
        self.add_domains_to_blacklist()

    def add_domains_to_blacklist(self):
        logging.info("Start adding domains to black list")
        self.order_id_to_domains = ParseDomains(self.run_yql_query(GetDomainsRequest(self.malware_orders)))
        logging.info('Domains parsed')
        new_black_list_rows = MakeNewDomainRows(self.malware_orders, self.order_id_to_domains, self.Parameters.google_sspid, self.unix_current_time, [int(x) for x in self.Parameters.good_domains.split(",")])
        cluster = 'markov'
        ytc = self.yt_connect(cluster)
        if ytc and len(new_black_list_rows):
            if self.Parameters.test_task:
                logging.info('Insert rows to domain black list {}'.format(new_black_list_rows))
            else:
                ytc.insert_rows(self.Parameters.domain_black_list_path, new_black_list_rows)
                self.solomon_data['blocked_domains_count'] = len(new_black_list_rows)
            logging.info('Inserted {} rows'.format(len(new_black_list_rows)))
        logging.info("Domains have been added successfully! Count {}, bad_orders: {}".format(len(new_black_list_rows), len(self.malware_orders)))

    def send_data_to_history_table(self):
        ytc = self.yt_connect(self.Parameters.history_cluster)
        logging.info("History len {}".format(len(self.new_history_rows)))
        for batch_no in range((len(self.new_history_rows) // BATCH_SIZE) + 1):
            left = batch_no * BATCH_SIZE
            right = min((batch_no + 1) * BATCH_SIZE, len(self.new_history_rows))
            try:
                ytc.insert_rows(self.Parameters.creatives_hist_tb_path, self.new_history_rows[left:right])
                logging.info("Succefully insert {0} rows".format((left, right)))
            except:
                logging.info("Fail insert rows")
                pass

    def get_strict_bad_orders(self):
        strict_bad_orders = set([])
        malware_orders = set([])

        for feedback in self.feedback_stats:
            creative_id = feedback[0]  # buyer_creative_id
            if 'domain' in creative_id:
                self.add_history_rows(creative_id, order_id, 'second account', 'SECOND_ACCOUNT')
                continue
            if len(creative_id.split('_')) < 2 :
                self.add_history_rows(creative_id, order_id, 'bad creative id', 'BAD')
                continue
            order_id = int(creative_id.split('_')[1])
            if order_id in self.black_list:
                self.add_history_rows(creative_id, order_id, 'order in black list', 'BAN')
                continue

            creative_status_code, event_count = int(feedback[1]), int(feedback[2])
            self.create_order_stat(order_id)
            self.all_orders_stats[order_id]['creatives'].add(creative_id)
            self.all_orders_stats[order_id]['mistakes'][creative_status_code] = self.all_orders_stats[order_id]['mistakes'].get(creative_status_code, 0) + event_count
            self.all_orders_stats[order_id]['all_bids'] += event_count

            ww_smart_orders = set([166979228,166732244,11189949,166965546,166979229,166999500, 167424964, 167424346, 167425206, 167763458])
            if (creative_status_code in STRICT_BAD_FEEDBACK_CODES or order_id in strict_bad_orders) and (order_id not in ww_smart_orders):
                strict_bad_orders.add(order_id)
                self.add_history_rows(creative_id, order_id, 'strict bad code', 'BAN')
            if creative_status_code == MALWARE_CODE:
                malware_orders.add(order_id)

        return strict_bad_orders, malware_orders

    def get_bad_moderate_orders(self):
        temporary_bad_orders = set([])
        for order_id in self.all_orders_stats.keys():
            if order_id in self.strict_bad_orders:
                continue
            if self.all_orders_stats[order_id]['all_bids'] < self.Parameters.min_bid_count:
                self.add_history_rows('', order_id, 'few bids', 'OK')
                continue
            not_moderate_bids = self.all_orders_stats[order_id]['mistakes'].get(RUSSIA_MODERATION_CODE, 0) + .0
            if not_moderate_bids / self.all_orders_stats[order_id]['all_bids'] < self.Parameters.bad_moderation_rate:
                self.add_history_rows('', order_id, '', 'OK')
            else:
                temporary_bad_orders.add(order_id)
                self.add_history_rows('', order_id, 'temporary ban for bad moderation in Russia', 'BAN')
        return temporary_bad_orders

    def fist_work_mode(self):
        self.black_list, self.orders_to_delete_from_blacklist = ParseBlackList(
            self.run_yql_query(GetBlackList(self.Parameters.black_list_table_path, self.Parameters.google_sspid)),
            self.unix_current_time - self.Parameters.time_from_delay * 60, self.Parameters.google_sspid
        )
        self.feedback_stats = self.run_yql_query(GetFeedbackStatistics(
            bs_ssp_feedback_table='//logs/bs-ssp-feedback-log/stream/5min',
            time_from=GetProcessTime(minutes=self.Parameters.time_from_delay),
            time_to=self.current_time,
            google_ssp_id=self.Parameters.google_sspid,
        ))

        self.strict_bad_orders, self.malware_orders = self.get_strict_bad_orders()
        self.temporary_bad_orders = self.get_bad_moderate_orders()
        self.clear_black_list_table()
        self.fill_black_list_table()

        self.solomon_data['deleted_orders_count'] = len(self.orders_to_delete_from_blacklist)
        self.solomon_data['strict_block_orders_count'] = len(self.strict_bad_orders)
        self.solomon_data['temporary_block_orders_count'] = len(self.temporary_bad_orders)

    def second_work_mode(self):
        self.strict_bad_orders, _ = ParseBlackList(self.run_yql_query(GetAllMalwareOrderByDomain(
            bs_ssp_feedback_table='//logs/bs-ssp-feedback-log/1h',
            bs_dsp_table='//logs/bs-dsp-log/1h',
            time_from=GetProcessTime(minutes=6 * 60),
            time_to=self.current_time,
            google_ssp_id=self.Parameters.google_sspid,
        )), 0, self.Parameters.google_sspid)

        self.fill_black_list_table()
        self.solomon_data['strict_block_orders_count_second_account'] = len(self.strict_bad_orders)

    def on_execute(self):
        self.new_history_rows = []
        self.crids_error = dict({})
        self.all_orders_stats = dict({})
        self.solomon_data = dict({})
        self.deleted_orders_count = 0
        self.malware_orders_by_domain = set([])
        self.strict_bad_orders = set([])
        self.malware_orders = set([])
        self.temporary_bad_orders = set([])

        self.yql_token = sdk2.task.Vault.data(self.Parameters.yql_token_owner, self.Parameters.yql_token_vault_name)

        self.unix_current_time = int(time.time())
        self.current_time = GetProcessTime()

        if self.Parameters.work_mode == FIRST_WORK_MODE:
            self.fist_work_mode()
        elif self.Parameters.work_mode == SECOND_WORK_MODE:
            self.second_work_mode()

        logging.info("Started YabsServerFillBlackListNew! time {0}".format(str(self.current_time)))
        SendDataToSolomon(self.solomon_data, self.Parameters.solomon_project, self.Parameters.solomon_cluster, self.Parameters.solomon_service, self.Parameters.solomon_key.data()['token'])
        self.send_data_to_history_table()
        logging.info("Finished YabsServerFillBlackListNew!")
