import requests
import time
import json
from datetime import datetime, timedelta

MODERATION_TIKET = 'MODERATION'

def GetProcessTime(frmt='%Y-%m-%dT%H:%M:%S', minutes=0):
    start_time = datetime.now() - timedelta(minutes=minutes)
    return start_time.strftime(frmt)


def SendDataToSolomon(data, solomon_project, solomon_cluster, solomon_service, solomon_token):
    SOLOMON_URL = 'https://solomon.yandex.net/api/v2/push?project={project}&cluster={cluster}&service={service}'.format(
        project=solomon_project,
        cluster=solomon_cluster,
        service=solomon_service
    )
    headers = {
        'Content-type': 'application/json',
        'Authorization': 'OAuth {token}'.format(token=solomon_token)
    }
    data = {
        "sensors": [{
            "labels": {"GoogleCreativeStatistic": "StrictBlockOrdersCountGoogleSSPSecondAccount"},
            "ts": int(time.time()),
            "value": int(data.get('strict_block_orders_count_second_account', 0)),
        }, {
            "labels": {"GoogleCreativeStatistic": "StrictBlockOrdersCountGoogleSSP"},
            "ts": int(time.time()),
            "value": int(data.get('strict_block_orders_count', 0)),
        },{
            "labels": {"GoogleCreativeStatistic": "TemporaryBlockOrdersCountGoogleSSP"},
            "ts": int(time.time()),
            "value": int(data.get('temporary_block_orders_count', 0)),
        },{
            "labels": {"GoogleCreativeStatistic": "DeletedFromBlacklistOrdersCountGoogleSSP"},
            "ts": int(time.time()),
            "value": int(data.get('deleted_orders_count', 0)),
        },{
            "labels": {"GoogleCreativeStatistic": "BlockDomainsCountGoogleSSP"},
            "ts": int(time.time()),
            "value": int(data.get('blocked_domains_count', 0)),
        }]
    }
    response = requests.post(SOLOMON_URL, data=json.dumps(data), headers=headers)
    response.raise_for_status()


def MakeNewBlackListRows(strict_bad_orders, temporary_bad_orders, sspid, unix_current_time):
    def make_new_raw(sspid, order_id, unix_current_time, tiket, reasons):
        return {
            'SSPID': sspid,
            'OrderID': int(order_id),
            'UpdateTime': unix_current_time,
            'Ticket': tiket,
            'FilterReasons': reasons,
            'FilteredCreatives': '',
            'GoodCreatives': '',
        }

    new_black_list_rows = []
    for order_id in strict_bad_orders:
        new_row = make_new_raw(sspid, order_id, unix_current_time, 'STRICT', 'strict reasons')
        new_black_list_rows.append(new_row)
    for order_id in temporary_bad_orders:
        new_row = make_new_raw(sspid, order_id, unix_current_time, MODERATION_TIKET, 'too much unmoderated bids')
        new_black_list_rows.append(new_row)
    return new_black_list_rows


def MakeNewDomainRows(strict_bad_orders, order_id_to_domains, sspid, unix_current_time, good_domains):
    def make_new_raw(sspid, domain_id, unix_current_time, reasons):
        return {
            'SSPID': sspid,
            'DomainID': int(domain_id),
            'UpdateTime': unix_current_time,
            'FilterReasons': reasons,
            'Tag': 'Auto'
        }

    new_black_list_rows = []
    for order_id in strict_bad_orders:
        for domain_id in order_id_to_domains[order_id]:
            if domain_id in good_domains:
                continue
            new_row = make_new_raw(sspid, domain_id, unix_current_time, 'strict reasons')
            new_black_list_rows.append(new_row)
    return new_black_list_rows


def ParseBlackList(json_black_list, time_from, sspid):
    black_list = []
    orders_to_delete_from_blacklist = []

    for raw in json_black_list:
        order_id = int(raw[0]) if raw[0] is not None else 0
        if order_id == 0: continue
        black_list.append(order_id)

        if len(raw) < 3: continue
        tiket = raw[1] if len(raw) < 2 or raw[1] is not None else ''
        updatetime = int(raw[2]) if len(raw) < 3 or raw[2] is not None else 0
        if tiket == MODERATION_TIKET and updatetime < time_from:
            orders_to_delete_from_blacklist.append({'OrderID': order_id, 'SSPID': sspid})
    return set(black_list), orders_to_delete_from_blacklist


def ParseDomains(json_domains):
    order_id_to_domains = {}

    for raw in json_domains:
        order_id = int(raw[0]) if raw[0] is not None else 0
        domain_id = int(raw[1]) if raw[1] is not None else 0
        if order_id == 0 or domain_id == 0:
            continue

        domains = order_id_to_domains.get(order_id, set([]))
        domains.add(domain_id)
        order_id_to_domains[order_id] = domains
    return order_id_to_domains
