#!/usr/bin/env python
# -*- coding: utf-8 -*-
from sandbox.common.errors import TaskFailure
import logging
from sandbox import sdk2
import time
from .yql_requests import GetTopMediaOrdersRequest, GetWhiteListRequest, GetHeavyCreativeIDList

from sandbox.projects.yabs.ssp.base_fill_whitelist.YabsServerFillWhiteList import BaseYabsServerFillWhiteList


class YabsServerFillGoogleWhiteList(BaseYabsServerFillWhiteList, sdk2.Task):
    class Parameters(BaseYabsServerFillWhiteList.Parameters):
        with BaseYabsServerFillWhiteList.Parameters.common_params() as common_params:
            sspid = sdk2.parameters.Integer("Google SSPID", default=17298340, required=True)
            white_ticket = sdk2.parameters.String("Google WhiteScript Task Ticket", default="BSSERVER-3728", required=True)
            send_detailed_skips = sdk2.parameters.Bool("Send detailed skip counters to Solomon", default=True, required=True)
            skip_already_added_orders = sdk2.parameters.Bool("Skip orders which have already been added to white lists", default=True, required=True)

        with BaseYabsServerFillWhiteList.Parameters.yql_params() as yql_params:
            yql_token_owner = sdk2.parameters.String("YQL Token Owner Name", default="robot-yabs-google", required=True)
            retry_period = sdk2.parameters.Integer("Time period to recheck yql completeness (seconds)", default=60 * 5)

        with BaseYabsServerFillWhiteList.Parameters.table_params() as table_params:
            order_info_table_path = sdk2.parameters.String("OrderInfo Table Path ", default="//home/yabs/dict/CaesarOrderInfo", required=True)

        with BaseYabsServerFillWhiteList.Parameters.order_limits_params() as order_limits_params:
            add_limit = sdk2.parameters.Integer("Max Count of new orders to white list", default=30, required=True)
            select_reach_orders_separately = sdk2.parameters.Bool("Select reach orders separately", default=False, required=True)
            with select_reach_orders_separately.value[True]:
                reach_percent = sdk2.parameters.Integer("Reach orders percent", default=50, required=True)

    class Context(BaseYabsServerFillWhiteList.Context):
        yql_requests = []

    def get_yql_results(self, query_name):
        results = super(YabsServerFillGoogleWhiteList, self).get_yql_results(query_name)
        return [int(next(iter(row.values()))) for row in results]

    def wait_yql(self, query_name):
        from yql.client.operation import YqlOperationStatusRequest

        status = YqlOperationStatusRequest(self.Context.yql_ids[query_name])
        status.run()
        if status.status in status.IN_PROGRESS_STATUSES:
            logging.info("yql query still running [{}]".format(query_name))
            raise sdk2.WaitTime(self.Parameters.retry_period)
        if not status.is_success:
            if status.errors:
                logging.error("YQL query [{}] returned errors:".format(query_name))
                for error in status.errors:
                    logging.error("- %s", str(error))
            raise TaskFailure("YQL query [{}] failed".format(query_name))

    def get_yql_requests(self, heavy_creatives):
        place_id_list = [542] if self.Parameters.select_reach_orders_separately else [542, 1542]
        place_id = ",".join(["'" + str(place_id) + "'" for place_id in place_id_list])
        heavy_ids = ",".join(["'"+str(crid)+"'" for crid in heavy_creatives])
        return [
            {
                'name': 'top',
                'query': GetTopMediaOrdersRequest(self.Context.bs_chevent_log_table, self.Parameters.order_info_table_path,
                                                  self.Parameters.wt_table_path, place_id,
                                                  heavy_ids if not self.Parameters.select_reach_orders_separately else None,
                                                  self.Parameters.sspid, self.Parameters.top_count,
                                                  self.Parameters.skip_already_added_orders),
            },
            {
                'name': 'top_1542',
                'query':  GetTopMediaOrdersRequest(self.Context.bs_chevent_log_table, self.Parameters.order_info_table_path,
                                                   self.Parameters.wt_table_path, "'1542'", heavy_ids,
                                                   self.Parameters.sspid, self.Parameters.top_count,
                                                   self.Parameters.skip_already_added_orders),
            } if self.Parameters.select_reach_orders_separately else None,
            {
                'name': 'wl',
                'query': GetWhiteListRequest(self.Parameters.wt_table_path, self.Parameters.sspid),
            },
        ]

    def get_data_from_yql(self):
        self.white_list = set(self.get_yql_results('wl'))
        self.top_media_orders = []

        if self.Parameters.select_reach_orders_separately:
            self.top_media_orders_542 = set(self.get_yql_results('top'))
            self.top_media_orders_1542 = set(self.get_yql_results('top_1542'))
            cnt = 0
            for row in self.top_media_orders_1542:
                if cnt <= int(self.Parameters.top_count * (self.Parameters.reach_percent / 100.0)):
                    self.top_media_orders.append(row)
                    cnt += 1
                else:
                    break

            logging.info("Selected {cnt} reach orders".format(cnt=cnt))

            for row in self.top_media_orders_542:
                if cnt <= self.Parameters.top_count:
                    self.top_media_orders.append(row)
                    cnt += 1
                else:
                    break
        else:
            self.top_media_orders = self.get_yql_results('top')
        self.top_media_orders = set(self.top_media_orders)

    def get_top(self):
        self.new_orders = []
        skip_cnt = skip_wl = 0
        for order_id in self.top_media_orders:
            if len(self.new_orders) >= self.Parameters.add_limit:
                break
            if order_id not in self.white_list:
                self.new_orders.append(order_id)
            else:
                skip_cnt += 1
                if self.Parameters.send_detailed_skips:
                    skip_wl += 1
        self.skipped_orders = {
            'total': skip_cnt,
            'wl': skip_wl,
        }
        logging.info("skip count {}".format(skip_cnt))

    def fill_white_list(self):
        self.get_top()
        self.write_white_list()

    def get_solomon_data(self):
        current_time = int(time.time())
        return {
            "sensors": [{
                "labels": {"GoogleWhiteListStatistics": "AddedOrdersHahn"},
                "ts": current_time,
                "value": int(self.added_orders_by_cluster['hahn']),
            }, {
                "labels": {"GoogleWhiteListStatistics": "AddedOrdersArnold"},
                "ts": current_time,
                "value": int(self.added_orders_by_cluster['arnold']),
            }, {
                "labels": {"GoogleWhiteListStatistics": "SkippedOrdersCount"},
                "ts": current_time,
                "value": int(self.skipped_orders['total']),
            }, {
                "labels": {"GoogleWhiteListStatistics": "SkippedOrdersWhiteList"},
                "ts": current_time,
                "value": int(self.skipped_orders['wl']),
            }]
        }

    def run_queries(self):
        with self.memoize_stage.get_heavy_creatives(commit_on_entrance=False, commit_on_wait=False):
            logging.info('Trying to get heavy creatives')
            query_name = 'heavy'
            with self.memoize_stage.run_heavy_creatives_yql:
                logging.info('Running heavy creatives yql...')
                query = GetHeavyCreativeIDList(100000)
                self.run_yql_query(query, self.yql_token, self.Parameters.priority_cluster, query_name)
            self.wait_yql(query_name)
            self.heavy_creatives = self.get_yql_results(query_name)

        with self.memoize_stage.get_yql_requests:
            logging.info('Getting requests...')
            self.Context.yql_requests = self.get_yql_requests(self.heavy_creatives)

        with self.memoize_stage.run_yql_requests(commit_on_entrance=False, commit_on_wait=False):
            logging.info('Running yql requests')
            for req in self.Context.yql_requests:
                if req is None:
                    continue
                with self.memoize_stage['Processing ' + req['name']](commit_on_entrance=False, commit_on_wait=False):
                    logging.info('Processing ' + req['name'])
                    with self.memoize_stage[req['name']]:
                        self.run_yql_query(req['query'], self.yql_token, self.Parameters.priority_cluster, req['name'])
                    self.wait_yql(req['name'])


class YabsServerFillWhiteList(YabsServerFillGoogleWhiteList):
    pass
