from sandbox import sdk2, common
from sandbox.sandboxsdk import environments

import logging

ADEXCHANGEBUYER_VERSIONS = ('v1.2', 'v1.3', 'v1.4')
ADEXCHANGEBUYERII_VERSIONS = ('v2beta1',)
ADEXCHANGEBUYERII_SERVICE_NAME = 'adexchangebuyer.googleapis.com'
DEFAULT_VERSION = 'v2beta1'
SECOND_ACCOUNT_QUERY = '&account-id=2&partner-stat-id=2&banner-lang=en'
NO_PROXY_URL_METADSP = 'https://bs-metadsp.yandex.ru/metadsp/181147?target-ref=https%3A%2F%2Fdsp.yandex.ru%2F&imp-id=1&ssp-id=17298340&partner-stat-id=10'
NO_PROXY_URL_BS = 'https://bs.yandex.ru/metadsp/181147?target-ref=https%3A%2F%2Fdsp.yandex.ru%2F&imp-id=1&ssp-id=17298340&partner-stat-id=10'

PROXY_DEFAULTS = {
    2: 10,
    5: 7000,
    6: 7000,
}

RELEASE_PROXY_DEFAULTS = {
    2: 10,
    5: 3000,
    6: 3000,
}

ACCOUNT_IDS = {
    '1': 198353918,
    '2': 378918135,
}


class YabsServerSSPGoogleApi(sdk2.Task):
    class Requirements(sdk2.Task.Requirements):
        environments = (
            environments.PipEnvironment('google-api-python-client', version='1.7.4', custom_parameters=['rsa==3.4.2']),
            environments.PipEnvironment('oauth2client', custom_parameters=['rsa==3.4.2']),
        )

    class Parameters(sdk2.Task.Parameters):
        key = sdk2.parameters.YavSecret("Google Key", default="sec-01e72kx10npmzw37j328yaqz6w", required=True)
        with sdk2.parameters.RadioGroup("Account Type", required=True, description='First or second google account') as account:
            account.values['1'] = account.Value(value='1', default=True)
            account.values['2'] = account.Value(value='2')

        with sdk2.parameters.RadioGroup('Preset', required=True, description='Pre-release is used to crop traffic before releases\n100%/50% are fixed values, not realtime') as preset:
            preset.values['pre-release'] = preset.Value(value='Pre-Release')
            preset.values['full'] = preset.Value(value='100%')
            preset.values['half'] = preset.Value(value='50%')
            preset.values['manual'] = preset.Value(value='manual', default=True)

            with preset.value['manual']:
                google_2_qps = sdk2.parameters.Integer("Maximum QPS (google2)", default=PROXY_DEFAULTS[2], required=True)
                google_5_qps = sdk2.parameters.Integer("Maximum QPS (google5)", default=PROXY_DEFAULTS[5], required=True)
                google_6_qps = sdk2.parameters.Integer("Maximum QPS (google6)", default=PROXY_DEFAULTS[6], required=True)

        test_mode = sdk2.parameters.Bool("Run without changing settings", default=False, required=True)
        with sdk2.parameters.Group('Proxy parameters') as proxy_params:
            do_not_use_2_proxy = sdk2.parameters.Bool("Don't use 2 proxy", default=False, required=True)
            do_not_use_5_proxy = sdk2.parameters.Bool("Don't use 5 proxy", default=False, required=True)
            do_not_use_6_proxy = sdk2.parameters.Bool("Don't use 6 proxy", default=False, required=True)

	with sdk2.parameters.RadioGroup("Cluster Type (no proxy mode)", required=True, description='') as bs_cluster:
            bs_cluster.values['bs-metadsp'] = bs_cluster.Value(value='bs-metadsp', default=True)
            bs_cluster.values['bs'] = bs_cluster.Value(value='bs')

        with sdk2.parameters.Output:
            prev_2_qps = sdk2.parameters.Integer("Previous QPS (google2)", default=-1, required=True)
            prev_5_qps = sdk2.parameters.Integer("Previous QPS (google2)", default=-1, required=True)
            prev_6_qps = sdk2.parameters.Integer("Previous QPS (google2)", default=-1, required=True)

    def get_service(self, version=DEFAULT_VERSION):
        import urllib2
        import httplib2
        from oauth2client.service_account import ServiceAccountCredentials
        from googleapiclient.discovery import build
        from googleapiclient.discovery import build_from_document

        credentials = ServiceAccountCredentials.from_json_keyfile_dict(self.Parameters.key.data(), ['https://www.googleapis.com/auth/adexchange.buyer'])
        http = credentials.authorize(httplib2.Http())
        if version in ADEXCHANGEBUYER_VERSIONS:
            service = build('adexchangebuyer', version, http=http)
        elif version in ADEXCHANGEBUYERII_VERSIONS:
            discovery_url = ('https://%s/$discovery/rest?version=%s' % (ADEXCHANGEBUYERII_SERVICE_NAME, version))
            discovery_doc = urllib2.urlopen(discovery_url).read()
            service = build_from_document(service=discovery_doc, http=http)
        else:
            raise ValueError('Invalid version provided. Supported versions are: %s' % ', '.join(
                ADEXCHANGEBUYER_VERSIONS + ADEXCHANGEBUYERII_VERSIONS))
        return service

    def get_proxy_qps(self):
        if self.Parameters.preset == 'pre-release':
            return RELEASE_PROXY_DEFAULTS
        if self.Parameters.preset == 'half':
            return {
                2: PROXY_DEFAULTS[2] // 2,
                5: PROXY_DEFAULTS[5] // 2,
                6: PROXY_DEFAULTS[6] // 2,
            }
        if self.Parameters.preset == 'manual':
            return {
                2: self.Parameters.google_2_qps,
                5: self.Parameters.google_5_qps,
                6: self.Parameters.google_6_qps,
            }
        return PROXY_DEFAULTS

    def patch_account(self, ad_exchange_buyer, account_id, body):
        import pprint
        from googleapiclient.errors import HttpError

        try:
            account = ad_exchange_buyer.accounts().patch(
                id=account_id,
                body=body).execute()
            logging.info('Patched Account "%d".' % account_id)
            logging.info(pprint.pformat(account))
        except HttpError as e:
            raise common.errors.TaskError(e)

    def list_account(self, ad_exchange_buyer):
        import pprint
        from googleapiclient.errors import HttpError

        try:
            accounts = ad_exchange_buyer.accounts().list().execute()
            if 'items' in accounts:
                stat = 'Found the following accounts for current user:\n'
                for account in accounts['items']:
                    stat += pprint.pformat(account) + '\n'
            else:
                raise common.errors.TaskError('No accounts found.')
            logging.info(stat)
            return accounts['items']
        except HttpError as e:
            raise common.errors.TaskError(e)

    def check_settings(self, proxy_no, before, after):
        if before != after:
            return '{proxy}: {before} -> {after}\n'.format(proxy=proxy_no, before=before, after=after)
        else:
            return ''

    def update_prev_qps(self, account, qps):
        changed_settings = ''
        for proxy in account['bidderLocation']:
            if proxy['url'].startswith('https://google2'):
                self.Parameters.prev_2_qps = proxy['maximumQps']
                changed_settings += self.check_settings(2, self.Parameters.prev_2_qps, qps[2])
            elif proxy['url'].startswith('https://google5'):
                self.Parameters.prev_5_qps = proxy['maximumQps']
                changed_settings += self.check_settings(5, self.Parameters.prev_5_qps, qps[5])
            elif proxy['url'].startswith('https://google6'):
                self.Parameters.prev_6_qps = proxy['maximumQps']
                changed_settings += self.check_settings(6, self.Parameters.prev_6_qps, qps[6])
        return changed_settings

    def get_proxy_urls(self):

        NO_PROXY_URL = ''
	if self.Parameters.bs_cluster == 'bs':
            NO_PROXY_URL = NO_PROXY_URL_BS
        else:
            NO_PROXY_URL = NO_PROXY_URL_METADSP

        return {
            2: u'https://google2.dsp.yandexadexchange.net/metadsp/181147?target-ref=https%3A%2F%2Fdsp.yandex.ru%2F&imp-id=1&ssp-id=17298340' if not self.Parameters.do_not_use_2_proxy else NO_PROXY_URL,
            5: u'https://google5-fin.dsp.yandexadexchange.net/metadsp/181147?target-ref=https%3A%2F%2Fdsp.yandex.ru%2F&imp-id=1&ssp-id=17298340' if not self.Parameters.do_not_use_5_proxy else NO_PROXY_URL,
            6: u'https://google6.dsp.yandexadexchange.net/metadsp/181147?target-ref=https%3A%2F%2Fdsp.yandex.ru%2F&imp-id=1&ssp-id=17298340' if not self.Parameters.do_not_use_6_proxy else NO_PROXY_URL,
        }

    def on_execute(self):
        service = self.get_service()
        qps = self.get_proxy_qps()
        account_id = ACCOUNT_IDS[self.Parameters.account]
        additional_query = SECOND_ACCOUNT_QUERY if self.Parameters.account == '2' else ''
        urls = self.get_proxy_urls()
        BODY = {
            'accountId': account_id,
            'bidderLocation': [
                {
                    'bidProtocol': u'PROTOCOL_OPENRTB_2_5',
                    'maximumQps': qps[2],
                    'region': u'EUROPE',
                    'url': urls[2] + additional_query
                }, {
                    'bidProtocol': u'PROTOCOL_OPENRTB_2_4',
                    'maximumQps': qps[5],
                    'region': u'EUROPE',
                    'url': urls[5] + additional_query
                }, {
                    'bidProtocol': u'PROTOCOL_OPENRTB_2_4',
                    'maximumQps': qps[6],
                    'region': u'EUROPE',
                    'url': urls[6] + additional_query
                }
            ]
        }
        logging.info('Previous state of accounts')
        accounts = self.list_account(service)
        changed_settings = ''
        for account in accounts:
            if account['id'] == account_id:
                changed_settings = self.update_prev_qps(account, qps)

        if not self.Parameters.test_mode:
            self.patch_account(service, BODY['accountId'], BODY)
            self.server.notification(
                body='Changed settings for Google\nCreated by: {owner}\nLink: https://sandbox.yandex-team.ru/task/{id}\n'
                     'Account: {account_id}\nDiff:\n{changed_settings}'.format(account_id=account_id, changed_settings=changed_settings,
                                                                               owner=self.author, id=self.id),
                recipients=['yury_zh', 'vbryadov', 'yanush00'],
                transport=common.types.notification.Transport.TELEGRAM
            )

