#!/usr/bin/env python
# encoding=utf-8

from sandbox import sdk2, common
from sandbox.sdk2 import service_resources
from sandbox.sandboxsdk import environments
from sandbox.projects.common.binary_task import LastBinaryTaskRelease, LastBinaryReleaseParameters
from sandbox.common.types import misc as ctm
import logging

REALTIMEBIDDING_API = 'https://www.googleapis.com/auth/realtime-bidding'
REALTIMEBIDDING_SERVICE = 'realtimebidding'
REALTIMEBIDDING_SERVICE_DISCOVERY = 'realtimebidding.googleapis.com'
REALTIMEBIDDING_VERSION = 'v1'
REALTIMEBIDDING_SCOPE = 'https://www.googleapis.com/auth/realtime-bidding'
UPLOAD_API_URL = 'https://cm.g.doubleclick.net'

ACCOUNT_IDS = {
    '1': '198353918',
    '2': '378918135',
    '3': '765859854'
}

ACCOUNT_COOKIE_MATCH_ID = {
    '1': 'yandexru',
    '2': 'yandex_llc',
    '3': 'yandexcom'
}

DEFAULT_URL_RESTRICTION = {
    'restrictionType': 'CONTAINS',
    'url': 'www'
}  # Doesn't important because we don't add new users via cookie matching/pixel matching


def get_protobuf_id_type(id_type):
    import cookie_bulk_upload_pb2 as upload_types

    proto_type_mapping = {
        'GAID': upload_types.ANDROID_ADVERTISING_ID,
        'IDFA': upload_types.IDFA,
        'YUID': upload_types.PARTNER_PROVIDED_ID,
        'GUID': upload_types.GOOGLE_USER_ID,
    }
    if id_type not in proto_type_mapping:
        return None
    return proto_type_mapping[id_type]


def add_id(proto_req, user_id, user_id_type, list_id, is_delete):
    operation = proto_req.ops.add()
    operation.user_id = user_id
    operation.user_id_type = user_id_type
    operation.user_list_id = list_id
    operation.delete = is_delete


class YabsServerUserListHelper(LastBinaryTaskRelease, sdk2.Task):
    class Requirements(sdk2.Requirements):
        dns = ctm.DnsType.DNS64
        environments = (
            environments.PipEnvironment('yandex-yt'),
            environments.PipEnvironment('yql'),
        )

    class Parameters(LastBinaryReleaseParameters):
        kill_timeout = 8 * 60 * 60  # for long bigb queries

        with sdk2.parameters.Group('YT params') as yt_params:
            yql_token_owner = sdk2.parameters.String("YQL Token Owner Name", default="robot-yabs-google", required=True)
            yql_token_vault_name = sdk2.parameters.String("YQL Robot Token Vault Name", default="robot_google_yql_token", required=True)
            yt_cluster = sdk2.parameters.String("YT clusters for read", default="hahn", required=True)

        with sdk2.parameters.Group('Google params') as google_params:
            key = sdk2.parameters.YavSecret("Google Key", default="sec-01e72kx10npmzw37j328yaqz6w", required=True)

        with sdk2.parameters.RadioGroup("Account Type", required=True, description='Google account') as account:
            account.values['1'] = account.Value(value='1', default=True)
            account.values['2'] = account.Value(value='2')
            account.values['3'] = account.Value(value='3')

        list_name = sdk2.parameters.String("User list name", default="", required=True)
        with sdk2.parameters.RadioGroup("Action Type", required=True,
                                        description='Update or create user list') as action:
            action.values['create'] = action.Value(value='create')
            action.values['update'] = action.Value(value='update')

            with action.value['create']:
                list_desc = sdk2.parameters.String("User list description", default="", required=True)
                list_expiration = sdk2.parameters.Integer("Membership expiration in days, from 1 to 540",
                                                          default=540, required=True)

            with action.value['update']:
                with sdk2.parameters.RadioGroup("Update Type", description='How update user list') as update_action:
                    update_action.values['automatic_add'] = update_action.Value(value='automatic add')
                    update_action.values['manual_add'] = update_action.Value(value='manual add', default=True)
                    update_action.values['manual_delete'] = update_action.Value(value='manual delete')

                    with sdk2.parameters.RadioGroup("Use id type", description='Type of user id (for manual)') as user_id_type:
                        user_id_type.values['GAID'] = user_id_type.Value(value='GAID')
                        user_id_type.values['IDFA'] = user_id_type.Value(value='IDFA')
                        user_id_type.values['YUID'] = user_id_type.Value(value='YUID')
                        user_id_type.values['GUID'] = user_id_type.Value(value='GUID')

                        user_ids = sdk2.parameters.String("User ids (comma separated, for manual mode)", default="", required=False)

    def on_create(self):
        self.Requirements.tasks_resource = service_resources.SandboxTasksBinary.find(
            owner="YABS_SERVER_SANDBOX_TESTS",
            attrs={"task_type": "YABS_SERVER_USER_LIST_HELPER"},
        ).first()

    def get_service(self):
        import urllib2
        from oauth2client.service_account import ServiceAccountCredentials
        from googleapiclient.discovery import build_from_document

        credentials = ServiceAccountCredentials.from_json_keyfile_dict(self.Parameters.key.data(),
                                                                       [REALTIMEBIDDING_SCOPE])
        discovery_url = ('https://%s/$discovery/rest?version=%s' % (
            REALTIMEBIDDING_SERVICE_DISCOVERY, REALTIMEBIDDING_VERSION))
        discovery_doc = urllib2.urlopen(discovery_url).read()
        service = build_from_document(service=discovery_doc, credentials=credentials)
        return service

    def get_buyer_name(self):
        return "buyers/" + ACCOUNT_IDS[self.Parameters.account]

    def get_upload_api_url(self):
        return UPLOAD_API_URL + '/upload?nid=' + ACCOUNT_COOKIE_MATCH_ID[self.Parameters.account]

    def find_list_id_by_name(self, service):
        from googleapiclient.errors import HttpError
        import pprint

        next_page_token = None
        logging.info("User lists for account")
        while True:
            try:
                lists = service.buyers().userLists().list(parent=self.get_buyer_name(),
                                                          pageToken=next_page_token).execute()
                logging.info("List batch: " + pprint.pformat(lists))
            except HttpError as e:
                raise common.errors.TaskError(e)

            if 'userLists' not in lists:
                break
            for list in lists['userLists']:
                if list['displayName'] == self.Parameters.list_name:
                    return int(list['name'][list['name'].rfind('/') + 1:])

            if 'nextPageToken' in lists:
                next_page_token = lists['nextPageToken']
            else:
                break

        return None

    def add_new_list(self, service):
        from googleapiclient.errors import HttpError
        import pprint

        user_list = {
            'displayName': self.Parameters.list_name,
            'description': self.Parameters.list_desc,
            'membershipDurationDays': self.Parameters.list_expiration,
            'urlRestriction': DEFAULT_URL_RESTRICTION
        }

        logging.info("Creating new list")
        try:
            response = service.buyers().userLists().create(parent=self.get_buyer_name(), body=user_list).execute()
            logging.info("New list: " + pprint.pformat(response))
        except HttpError as e:
            raise common.errors.TaskError(e)

    def send_update_req(self, proto_req, send_notification=False, log=False):
        import cookie_bulk_upload_pb2 as upload_types
        from google.protobuf.json_format import MessageToJson
        import requests

        if send_notification:
            proto_req.send_notifications = True

        if log:
            logging.info("Request proto:\n" + str(MessageToJson(proto_req)))

        result = requests.post(
            self.get_upload_api_url(),
            headers={
                'Content-Type': 'application/x-protobuf'
            },
            data=proto_req.SerializeToString()
        )

        logging.info("Response code: " + str(result.status_code))
        response = upload_types.UpdateUsersDataResponse()
        try:
            response.ParseFromString(result.content)
            logging.info("Response:\n" + str(MessageToJson(response)))
        except:
            logging.error("Unable to parse google response")

    def upload_user_ids(self, list_id, user_ids, is_delete=False, bulk_size=2000):
        import cookie_bulk_upload_pb2 as upload_types

        cnt = 0
        full_cnt = 0
        update_request = upload_types.UpdateUsersDataRequest()
        for id_type, id in user_ids:
            proto_id_type = get_protobuf_id_type(id_type)
            if proto_id_type is None:
                continue

            add_id(update_request, id.strip(), proto_id_type, list_id, is_delete)
            cnt += 1
            full_cnt += 1
            if cnt >= bulk_size != -1:
                logging.info("Sending: " + str(cnt))
                self.send_update_req(update_request)
                cnt = 0
                update_request = upload_types.UpdateUsersDataRequest()
                logging.info("Total sent: " + str(full_cnt))

        if cnt != 0:
            logging.info("Sending: " + str(cnt))
            self.send_update_req(update_request, log=True)
            logging.info("Total sent: " + str(full_cnt))

    def run_yql_query(self, query_text):
        from yql.api.v1.client import YqlClient

        logging.info("Connecting to yql")
        client = YqlClient(db=self.Parameters.yt_cluster,
                           token=sdk2.task.Vault.data(self.Parameters.yql_token_owner,
                                                      self.Parameters.yql_token_vault_name))
        request = client.query(query_text)

        logging.info("Running yql query")
        request.run()
        request.wait_progress()

        if not request.is_success:
            logging.info('\n'.join([str(err) for err in request.errors]))
            raise common.errors.TaskError("YQL Error. Check logs")

        logging.info('Parsing yql result')
        result = []
        for table in request.get_results():
            table.fetch_full_data()
            for row in table.rows:
                result.append(row)
        logging.info("Finish yql success, collect {} rows".format(len(result)))
        return result

    def get_yql_ids(self):
        import yql_requests
        logging.info("Yql query: " + yql_requests.get_profile_uniq_ids_query())
        return self.run_yql_query(yql_requests.get_profile_uniq_ids_query())

    def get_ids_from_params(self):
        return [(self.Parameters.user_id_type, id) for id in self.Parameters.user_ids.split(",")]

    def update_user_list(self, list_id):
        delete = False
        automatic_update = False

        if self.Parameters.update_action == 'manual_delete':
            delete = True
        if self.Parameters.update_action == 'automatic_add':
            automatic_update = True

        if not automatic_update:
            user_ids = self.get_ids_from_params()
        else:
            user_ids = self.get_yql_ids()

        self.upload_user_ids(list_id, user_ids, is_delete=delete)

    def on_execute(self):
        service = self.get_service()
        list_id = self.find_list_id_by_name(service)

        if self.Parameters.action == 'create':
            if list_id is not None:
                raise common.errors.TaskError("Invalid user list name. User list with this name already exists")
            self.add_new_list(service)

        if self.Parameters.action == 'update':
            if list_id is None:
                raise common.errors.TaskError("Invalid user list name. User list doesn't exists")
            self.update_user_list(list_id)
