import logging
import os
import subprocess
import time
import urllib2

from sandbox import sdk2
from sandbox.common.fs import get_unique_file_name
from sandbox.projects.common.arcadia import sdk
from sandbox.projects.common.yabs.cachedaemon import CacheDaemonStubSandboxNonIntegrated
from sandbox.projects.yabs.qa.module_base import ModuleBase

PORT_NAMES = ['renderer_admin_port', 'apphost_admin_port']


class SsrSUT(ModuleBase):
    def __init__(self, adapter):
        ModuleBase.__init__(self, adapter)
        self._active = False

        self._ports = self.adapter.get_ports_map()
        self._ports['apphost_port'] = self._ports['apphost_admin_port'] + 4

        self._tokens = self.adapter.get_tokens()
        os.environ.update(self._tokens)

        self._cachedaemon = CacheDaemonStubSandboxNonIntegrated(
            cache_daemon_executable_path=self.adapter.get_cache_daemon_executable_path(),
            dump_path=self.adapter.get_cache_daemon_stub_path(),
            data_dir=get_unique_file_name(self.adapter.get_work_dir(), 'cache_daemon_ssr_data'),
            log_subdir=get_unique_file_name(self.adapter.get_logs_dir(), 'cache_daemon_ssr_logs'),
            start_on_creation=False,
            key_header=self.adapter.get_cache_daemon_key_headers(),
            services=None,
            services_ports=self._ports.get('cache_daemon_ports'),
        )

        self._mount_arc()
        self._build_apphost()
        self._unpack_renderer()

    def __del__(self):
        if self.local_arcadia is not None:
            self.local_arcadia.__del__()

    def _mount_arc(self):
        logging.info("Getting local arcadia")
        self.local_arcadia = sdk.mount_arc_path('arcadia-arc:/#trunk')
        self.local_arcadia_path = self.local_arcadia.__enter__()

        logging.info("Getting 'ya' path")
        self.ya_path = os.path.join(self.local_arcadia_path, 'ya')
        if not os.path.exists(self.ya_path):
            self.ya_path = os.path.join(self.local_arcadia_path, 'devtools', 'ya', 'ya')

    def _build_apphost(self):
        # build with -D param (download latest released binaries)
        cmd = [
            self.ya_path,
            'tool',
            'apphost',
            'setup',
            '-y',
            '-D',
            '--local-arcadia-path',
            self.local_arcadia_path,
            self.adapter.get_apphost_mode(),
            '--vertical',
            'PCODE',
        ]
        logging.info('Build dir and download binaries from Sandbox: {}'.format(cmd))

        self._apphost_build_log_context = sdk2.helpers.ProcessLog(self.adapter.task_instance, 'apphost.build')
        self._apphost_build_log_context.__enter__()
        subprocess.Popen(
            cmd, stdout=self._apphost_build_log_context.stdout, stderr=self._apphost_build_log_context.stderr
        )

    def _unpack_renderer(self):
        self._pcode_unpacked_path = os.path.join(self.adapter.get_work_dir(), 'renderer')

        logging.info('Unpacking pcode resource to {}'.format(self._pcode_unpacked_path))
        with sdk2.helpers.ProcessLog(self.adapter.task_instance, 'renderer.unpack') as process_log:
            subprocess.Popen(
                ['tar', '-xvf', self.adapter.get_pcode_resource_path(), '-C', self.adapter.get_work_dir()],
                stdout=process_log.stdout,
                stderr=process_log.stderr,
            ).wait()

    def _run_renderer(self):
        os.environ['PATH'] += os.pathsep + os.path.join(self._pcode_unpacked_path, self.adapter.get_node_relpath())

        root_dir = os.getcwd()
        os.chdir(os.path.join(self._pcode_unpacked_path, 'arcadia/adv/pcode/web/pcode'))

        cmd = [
            'npx',
            'archon',
            'pcode-renderer',
            '--rendererPort',
            str(self._ports['renderer_port']),
            '--renderer-devops-port',
            str(self._ports['renderer_admin_port']),
            '--pcode-resource-id',
            str(self.adapter.get_web_pcode_micro_package_id()),
        ]
        logging.info('Running renderer with command: {}'.format(cmd))

        self._renderer_process_log_context = sdk2.helpers.ProcessLog(self.adapter.task_instance, 'renderer.run')
        self._renderer_process_log_context.__enter__()
        self._renderer_process = subprocess.Popen(
            cmd, stdout=self._renderer_process_log_context.stdout, stderr=self._renderer_process_log_context.stderr
        )
        os.chdir(root_dir)

    def _stop_renderer(self, *args):
        self._renderer_process.kill()
        self._renderer_process.communicate()
        self._renderer_process_log_context.__exit__(*args)
        self._renderer_process_log_context = None

    def _run_apphost(self):
        cmd = [
            self.ya_path,
            'tool',
            'apphost',
            'setup',
            '-y',
            '-S',
            '-D',
            '--local-arcadia-path',
            self.local_arcadia_path,
            '-P',
            str(self._ports['apphost_admin_port']),
            self.adapter.get_apphost_mode(),
            '--vertical',
            'PCODE',
        ]
        logging.info('Running apphost with command: {}'.format(cmd))

        self._apphost_process_log_context = sdk2.helpers.ProcessLog(self.adapter.task_instance, 'apphost.run')
        self._apphost_process_log_context.__enter__()
        self._apphost_process = subprocess.Popen(
            [self.ya_path, '-h'],
            stdout=self._apphost_process_log_context.stdout,
            stderr=self._apphost_process_log_context.stderr,
        )
        self._apphost_process = subprocess.Popen(
            cmd, stdout=self._apphost_process_log_context.stdout, stderr=self._apphost_process_log_context.stderr
        )

    def _stop_apphost(self, *args):
        self._apphost_process.kill()
        self._apphost_process.communicate()
        self._apphost_process_log_context.__exit__(*args)
        self._apphost_process_log_context = None

    def wait_ping(self, port, timeout=300):
        if not self._active:
            raise RuntimeError('wait_ping was used to check on inactive service')
        deadline = time.time() + timeout
        pause = 0.25
        request = urllib2.Request("http://localhost:{}/admin?action=ping".format(port))
        while True:
            error_msg = ''
            try:
                urllib2.urlopen(request, timeout=10)
            except urllib2.HTTPError as e:
                error_msg = "/admin?action=ping returned %d" % e.code
            except Exception as e:
                error_msg = "/admin?action=ping exception: %s" % (str(e))
            else:
                break

            now = time.time()
            if now > deadline:
                raise RuntimeError("No 200 response after %s seconds (port %s): %s" % (timeout, port, error_msg))
            logging.info(error_msg)
            time.sleep(pause)
            pause *= 1.5
            logging.info("Retrying /admin?action=ping ...")

        logging.info("Obtained 200 response on port %s" % port)

    def __enter__(self):
        self._cachedaemon.__enter__()
        self._run_renderer()
        self._run_apphost()

        self._active = True
        logging.info('Ping apphost')
        self.wait_ping(self._ports['apphost_admin_port'],  timeout=20*60)
        logging.info('Ping renderer')
        self.wait_ping(self._ports['renderer_port'])

        return self

    def __exit__(self, *args):
        self._stop_apphost(*args)
        self._stop_renderer(*args)
        self._cachedaemon.__exit__(*args)
        self._active = False

    def get_port(self):
        if self._active:
            return self._ports['apphost_port']
        else:
            return None
