import gzip
import json
import logging
import os
import posixpath
import re
import textwrap

from sandbox import common, sdk2
from sandbox.projects.common.yabs.server.requestlog import iterate
from sandbox.projects.common.yabs.server.util.general import try_get_from_vault
from sandbox.projects.yabs.qa.resource_types import YABS_REPORT_RESOURCE, YABS_SERVER_REQUEST_LOG_GZ
from sandbox.projects.yabs.qa.tasks.YabsServerB2BFuncShootCmp import YabsServerB2BFuncShootCmp
from sandbox.projects.yabs.qa.tasks.YabsServerB2BFuncShootCmp.report import CmpReport
from sandbox.projects.yabs.qa.utils.resource import sync_resource
from sandbox.projects.yabs.ssr.resources import BsSsrDiffResult, BsSsrDiffReport, BsSsrHtmlDiffResult, BsSsrResponseDump, AdaptedMordaResponses
from sandbox.projects.yabs.ssr.util import run_command
from sandbox.sandboxsdk import environments

from compare import compare_results
from default_painting import (
    DEFAULT_BODY_SUBSTITUTES,
    DEFAULT_JSON_KEYS_TO_DELETE,
    DEFAULT_XML_KEYS_TO_DELETE,
    DEFAULT_HEADERS_PAINTING,
    DEFAULT_HTML_SUBSTITUTES,
    DEFAULT_CSS_SUBSTITUTES,
    DEFAULT_BASE64_PREFIXES,
    DEFAULT_HTML_TAGS,
    DEFAULT_HTML_TAGS_REMOVE_TO_CONVERT_XML,
    DEFAULT_FILTER_BY_PRE_HEADERS,
    DEFAULT_FILTER_BY_TEST_HEADERS,
    DEFAULT_JSON_PADDING_NAME,
)
from process import process_results


REPORT_DIR = 'report_dir'
HTML_DIFF_DIR = 'html_diff_dir'
CONTAINER_RESOURCE = 2044166647


class BsSsrCmpTask(sdk2.Task):
    """Diff task for SSR shoot results"""

    class Requirements(sdk2.Task.Requirements):
        environments = (
            environments.PipEnvironment('yandex-yt'),
            environments.PipEnvironment('yandex-yt-yson-bindings'),
            environments.PipEnvironment('yandex-yt-yson-bindings-skynet'),
        )
        container_resource = CONTAINER_RESOURCE

    class Context(sdk2.Task.Context):
        has_diff = True
        num_of_requests_with_diff = 0
        num_of_requests = 0

    class Parameters(sdk2.Task.Parameters):
        description = 'CMP task for results of SSR shoots.'

        with sdk2.parameters.Group('Response data') as response_data:
            with sdk2.parameters.RadioGroup('Response type', required=True, description='Type of response format') as response_type:
                response_type.values['ssr'] = response_type.Value(value='ssr', default=True)
                response_type.values['morda'] = response_type.Value(value='morda')

            with response_type.value['ssr']:
                pre_shoot_result = sdk2.parameters.Resource('Pre shoot result dump', resource_type=BsSsrResponseDump)
                test_shoot_result = sdk2.parameters.Resource('Test shoot result dump', resource_type=BsSsrResponseDump)
            with response_type.value['morda']:
                pre_shoot_result_morda = sdk2.parameters.Resource('Pre shoot result dump for Morda', resource_type=AdaptedMordaResponses)
                test_shoot_result_morda = sdk2.parameters.Resource('Test shoot result dump for Morda', resource_type=AdaptedMordaResponses)

        with sdk2.parameters.Group('Other resources') as resources:
            template_resource = sdk2.parameters.Resource('Template resource', resource_type=YABS_REPORT_RESOURCE)
            requestlog_resource = sdk2.parameters.Resource(
                'Requestlog resource', resource_type=YABS_SERVER_REQUEST_LOG_GZ
            )

        with sdk2.parameters.Group('Paintings') as painting_settings:
            headers_to_replace = sdk2.parameters.JSON('Headers painting', default=DEFAULT_HEADERS_PAINTING)
            body_substitutes = sdk2.parameters.JSON('Substitutes applied to entity', default=DEFAULT_BODY_SUBSTITUTES)
            json_keys_to_delete = sdk2.parameters.JSON(
                'List of keys to be deleted from json during comparison', default=DEFAULT_JSON_KEYS_TO_DELETE
            )
            xml_keys_to_delete = sdk2.parameters.JSON(
                'List of tags to be deleted from XML during comparison', default=DEFAULT_XML_KEYS_TO_DELETE
            )
            css_substitutes = sdk2.parameters.JSON(
                'Substitutes applied to decoded CSS', default=DEFAULT_CSS_SUBSTITUTES
            )
            html_substitutes = sdk2.parameters.JSON(
                'Substitutes applied to decoded HTML', default=DEFAULT_HTML_SUBSTITUTES
            )
            base64_prefixes = sdk2.parameters.JSON(
                'Prefixes followed with base64 string in commas to decode in entity', default=DEFAULT_BASE64_PREFIXES
            )
            html_tags = sdk2.parameters.JSON(
                'Fields in json to try interpret like HTML', default=DEFAULT_HTML_TAGS
            )
            html_tags_remove_to_convert_xml = sdk2.parameters.JSON(
                'HTML tags, which removed, to parse HTML as XML', default=DEFAULT_HTML_TAGS_REMOVE_TO_CONVERT_XML
            )
            filter_by_pre_headers = sdk2.parameters.JSON(
                'Filter diff by pre-headers (e.g. ["uniformat-product-type not-find Media"])',
                default=DEFAULT_FILTER_BY_PRE_HEADERS
            )
            filter_by_test_headers = sdk2.parameters.JSON(
                'Filter diff by test-headers (e.g. ["uniformat-product-type != None"])',
                default=DEFAULT_FILTER_BY_TEST_HEADERS
            )

        with sdk2.parameters.Group('Diff settings') as diff_settings:
            compare_bodies = sdk2.parameters.Bool('Compare full bodies', default=True)
            use_js_differ = sdk2.parameters.Bool('Use JS html-differ to compare html', default=False)
            html_diff_window_size = sdk2.parameters.Integer(
                'Number of context symbols added to each diff in html', default=100000
            )

        with sdk2.parameters.Group('YT settings') as yt_settings:
            yt_token = sdk2.parameters.String('YT token name in Sandbox vault', default='yabs-cs-sb-yt-token')

        with sdk2.parameters.Group('Common settings') as common_settings:
            key_header = sdk2.parameters.String('Key header name', default='X-Yabs-Ssr-Req-Id')
            ttl = sdk2.parameters.Integer('TTL for diff results', default=7)
            n_compare_jobs = sdk2.parameters.Integer('Number of threads for results comparison', default=8)
            n_process_jobs = sdk2.parameters.Integer('Number of threads for report processing', default=1)

        with sdk2.parameters.Output:
            report = sdk2.parameters.Resource('Report resource', resource_type=BsSsrDiffResult)
            html_diff = sdk2.parameters.Resource(
                'Packed HTML diff (available only with use_js_differ)', resource_type=BsSsrHtmlDiffResult
            )

    @staticmethod
    def read_yt_table(yt, table_path, key_column, columns=None):
        columns_to_read = [key_column] + columns if columns is not None else None
        result = {}
        for row in yt.read_table(yt.TablePath(table_path, columns=columns_to_read)):
            key_value = row[key_column]
            columns_value = {col: row[col] for col in columns} if columns is not None else row
            result[key_value] = columns_value
        return result

    def _unpack_request_log(self, resource):
        path = str(sdk2.ResourceData(resource).path)
        with gzip.GzipFile(filename=path, mode='rb', mtime=0) as gz:
            requests = [req for req, _ in iterate(gz, r'\s*')]
        request_ids = [re.search(r'{}: (\d*)'.format(self.Parameters.key_header), x).group(1) for x in requests]

        for i in range(len(requests)):
            try:
                json.dumps(requests[i])
            except:
                requests[i] = 'Failed to parse request. Look for it in request-data-log resource.'

        return dict(zip(request_ids, requests))

    @staticmethod
    def _get_file(filename):
        path = os.path.join(os.path.dirname(__file__), filename)

        if not common.import_hook.inside_the_binary():
            return path

        from library.python import resource
        logging.info('Downloading file: ', path)
        data = resource.find(path)
        with open(filename, 'w') as f:
            f.write(data)
        return os.path.join(os.getcwd(), filename)

    def _init_html_diff(self):
        if self.Parameters.use_js_differ:
            if not os.path.exists(HTML_DIFF_DIR):
                os.makedirs(HTML_DIFF_DIR)
            return os.path.join(os.getcwd(), HTML_DIFF_DIR)
        else:
            return None

    def _sync_report_resource(self):
        if not os.path.exists(REPORT_DIR):
            os.mkdir(REPORT_DIR)
            template_path = sync_resource(
                resource=self.Parameters.template_resource, resource_type=YABS_REPORT_RESOURCE
            )
            CmpReport(REPORT_DIR).prepare(template_path)

    def _create_report(self, pre_data, test_data, diff_data):
        diff_results = []
        pre_codes = set()
        test_codes = set()
        handlers = set()
        tags = set()

        logging.info('Unpaccking request log.')
        if self.Parameters.response_type == 'ssr':
            requests = self._unpack_request_log(self.Parameters.requestlog_resource)
        else:
            requests = {
                str(request_id): 'Stubbed request for morda'
                for request_id in pre_data.keys()
            }

        logging.info('Processing results.')
        results = process_results(
            requests,
            pre_data,
            test_data,
            diff_data,
            self.Parameters.n_process_jobs,
            REPORT_DIR,
        )

        self.Context.num_of_requests = len(results)

        for result in results:
            if result['has_diff']:
                self.Context.num_of_requests_with_diff += 1

            pre_codes.add(result['pre_code'])
            test_codes.add(result['test_code'])
            handlers.add(result['handler'])
            tags.update(result['tags'])

            diff_results.append(
                {
                    'status': ('failed' if result['has_diff'] else 'passed'),
                    'search': {
                        'pre.code': result['pre_code'],
                        'test.code': result['test_code'],
                        'handler': result['handler'],
                        'tags': list(result['tags']),
                    },
                    'name': str(int(result['test_id'])),
                    'id': int(result['test_id']),
                    'diffLinesCount': 20,
                }
            )

        metadata = {
            "Tests": len(diff_results),
            "Failures": self.Context.num_of_requests_with_diff,
        }

        report = {
            'search': {
                'pre.code': list(pre_codes),
                'test.code': list(test_codes),
                'handler': list(handlers),
                'tags': list(tags),
            },
            'meta': [{'title': title, 'value': value} for title, value in metadata.iteritems()],
            'results': diff_results,
        }

        report_file_path = os.path.join(REPORT_DIR, 'report.json')
        with open(report_file_path, 'w') as report_file:
            json.dump(report, report_file)

        logging.info(
            'Finished. {} out of {} requests has diff.'.format(
                self.Context.num_of_requests_with_diff, self.Context.num_of_requests
            )
        )

        self.Context.has_diff = bool(self.Context.num_of_requests_with_diff)

        report_resource = BsSsrDiffReport(self, 'Report resource', REPORT_DIR, ttl=self.Parameters.ttl)
        sdk2.ResourceData(report_resource).ready()

        report_url = YabsServerB2BFuncShootCmp.get_resource_url(self.id, REPORT_DIR, report_resource.id)
        diff_url = posixpath.join(report_url, 'index.html')

        execution_report = textwrap.dedent(
            '''\
            {failures} out of {total_tests} tests failed.
            <a href="{diff_url}" target="_blank">Diff viewer</a>
        '''
        ).format(
            failures=self.Context.num_of_requests_with_diff,
            total_tests=self.Context.num_of_requests,
            diff_url=diff_url,
        )

        self.set_info(execution_report, do_escape=False)

        with open('report.html', 'w') as file:
            file.write(execution_report.replace('\n', '<br>'))

        self.Parameters.report = BsSsrDiffResult(self, 'Report resource', 'report.html', ttl=self.Parameters.ttl)
        sdk2.ResourceData(self.Parameters.report).ready()

    def on_execute(self):
        import yt.wrapper as yt

        yt.config['proxy']['url'] = 'hahn'
        yt.config['token'] = try_get_from_vault(self, self.Parameters.yt_token)

        self._sync_report_resource()

        html_differ_path = self._get_file('html_differ.js')
        logging.info(html_differ_path)
        run_command('npm link html-differ')

        html_differ_output_path = self._init_html_diff()

        def read_data(yt_path):
            logging.info('Reading shoot results from path: {}'.format(yt_path))
            return self.read_yt_table(yt, yt_path, 'RequestID', ['HttpCode', 'Url', 'Data'])

        def read_json_data(path):
            with open(str(path), 'r') as f:
                return json.loads(f.read())

        if self.Parameters.response_type == 'ssr':
            pre_data = read_data(self.Parameters.pre_shoot_result.yt_table_path)
            test_data = read_data(self.Parameters.test_shoot_result.yt_table_path)
        else:
            pre_data = read_json_data(sync_resource(resource=self.Parameters.pre_shoot_result_morda, resource_type=AdaptedMordaResponses))
            test_data = read_json_data(sync_resource(resource=self.Parameters.test_shoot_result_morda, resource_type=AdaptedMordaResponses))

        logging.info('Comparing results.')
        diff_data = compare_results(
            pre_data,
            test_data,
            self.Parameters.n_compare_jobs,
            self.Parameters.headers_to_replace,
            self.Parameters.body_substitutes,
            self.Parameters.json_keys_to_delete,
            self.Parameters.xml_keys_to_delete,
            self.Parameters.css_substitutes,
            self.Parameters.html_substitutes,
            self.Parameters.compare_bodies,
            self.Parameters.html_diff_window_size,
            self.Parameters.base64_prefixes,
            self.Parameters.html_tags,
            self.Parameters.html_tags_remove_to_convert_xml,
            self.Parameters.filter_by_pre_headers,
            self.Parameters.filter_by_test_headers,
            DEFAULT_JSON_PADDING_NAME,
            html_differ_path,
            html_differ_output_path,
        )
        logging.info('Finished comparison.')

        if html_differ_output_path is not None:
            self.Parameters.html_diff = BsSsrHtmlDiffResult(
                self,
                'JS html-differ HTML diff for SSR tests',
                html_differ_output_path,
                ttl=self.Parameters.ttl,
            )

        self._create_report(pre_data, test_data, diff_data)
