import base64
import json
import os
import xml.dom.minidom as md

from sandbox.projects.yabs.base_differ.base_response_differ import BaseResponseDiffer, HeaderFilter
from sandbox.projects.yabs.ssr.util import run_command


class ResponseDiffer(BaseResponseDiffer):
    @staticmethod
    def _simple_html_diff(html_pre, html_test):
        diff = ''
        if html_pre != html_test:
            diff = '-' + html_pre + '\n+' + html_test + '\n'
        return diff

    def __init__(
        self,
        headers_to_replace,
        body_substitutes,
        json_keys_to_delete,
        base64_prefixes,
        xml_keys_to_delete,
        css_substitutes,
        html_substitutes,
        working_dir,
        compare_bodies,
        window_size,
        html_differ_path,
        html_differ_output_path,
        html_tags,
        html_tags_remove_to_convert_xml,
        filter_by_pre_headers,
        filter_by_test_headers,
        padding_name
    ):
        super(ResponseDiffer, self).__init__(headers_to_replace, body_substitutes, json_keys_to_delete, base64_prefixes)
        self.css_substitutes = css_substitutes
        self.html_substitutes = html_substitutes
        self.working_dir = working_dir
        self.compare_bodies = compare_bodies
        self.window_size = window_size
        self.html_differ_path = html_differ_path
        self.html_differ_output_path = html_differ_output_path
        self.xml_tags_to_delete = xml_keys_to_delete
        self.html_tags = set(html_tags)
        self.html_tags_remove_to_convert_xml = html_tags_remove_to_convert_xml
        self.filter_by_pre_headers = [HeaderFilter(filt) for filt in filter_by_pre_headers]
        self.filter_by_test_headers = [HeaderFilter(filt) for filt in filter_by_test_headers]
        self.padding_name = padding_name

        if not os.path.exists(self.working_dir):
            os.makedirs(self.working_dir)

        self.no_diff_response = {
            'HasDiff': False,
            'StatusDiff': '',
            'HeadersDiff': '',
            'EntityBase64': False,
            'EntitiesDiff': '',
            'CssDiff': '',
            'HtmlDiff': '',
        }

    def _js_html_diff(self, html_pre, html_test, output_path, working_dir, html_differ_path, window_size):
        pre_path = os.path.join(working_dir, 'pre.html')
        with open(pre_path, 'w') as f:
            f.write(html_pre)

        test_path = os.path.join(working_dir, 'test.html')
        with open(test_path, 'w') as f:
            f.write(html_test)

        run_command(
            'node {html_differ_path} {pre_path} {test_path} {output_path} {window_size}'.format(
                html_differ_path=html_differ_path,
                pre_path=pre_path,
                test_path=test_path,
                output_path=output_path,
                window_size=window_size,
            ),
            verbose=False,
        )

    def _html_diff(self, html_pre, html_test, html_substitutes, html_differ_output_path, working_dir, html_differ_path, window_size):
        html_pre = self.apply_substituttes(html_pre, html_substitutes)
        html_test = self.apply_substituttes(html_test, html_substitutes)
        html_diff = self._simple_html_diff(html_pre, html_test)
        if html_differ_output_path is not None and html_pre != html_test:
            self._js_html_diff(html_pre, html_test, html_differ_output_path, working_dir, html_differ_path, window_size)
        return html_diff

    def _entities_diff(self, entity_pre, entity_test, is_pre_json, is_test_json, is_pre_xml, is_test_xml):
        def modify_entity(entity, is_json, is_xml):
            if is_json:
                return [json.dumps(entity)]
            if is_xml:
                return [entity.toxml()]
            return [entity]

        if not self.compare_bodies:
            return ''

        if is_pre_json:
            entity_pre = self.json_painter.apply_json_paints(entity_pre)
        if is_test_json:
            entity_test = self.json_painter.apply_json_paints(entity_test)

        if is_pre_json and is_test_json:
            return self._dict_diff(entity_pre, entity_test)
        if is_pre_xml and is_test_xml:
            return self._xml_diff(entity_pre, entity_test)

        entity_pre = modify_entity(entity_pre, is_pre_json, is_pre_xml)
        entity_test = modify_entity(entity_test, is_test_json, is_test_xml)
        return self._string_lists_diff(entity_pre, entity_test)

    def _try_parse_xml_in_json(self, d):
        for field, value in (d.items() if isinstance(d, dict) else enumerate(d)):
            if isinstance(value, str) or isinstance(value, unicode):
                like_html = field in self.html_tags
                xml, is_xml = self._try_parse_xml(value, like_html)
                if is_xml:
                    d[field] = xml.toprettyxml(indent=' ').splitlines()
                else:
                    d[field] = self._try_get_json_inside(value)
            elif isinstance(value, dict) or isinstance(value, list):
                d[field] = self._try_parse_xml_in_json(value)
        return d

    def _try_to_parse_json(self, s):
        s, is_json = self.try_to_json(s)
        if is_json:
            self._try_parse_xml_in_json(s)
        return s, is_json

    def _preprocess_xml(self, node, like_html):  # return True if we want to remove this node
        if node.attributes:
            for attr_del, value_del in self.xml_tags_to_delete['remove-tag-by-attribute-value'].get(node.nodeName, dict()).items():
                if attr_del in node.attributes.keys() and node.attributes[attr_del].value == value_del:
                    return True
            for attr_del in self.xml_tags_to_delete['remove-attribute'].get(node.nodeName, []):
                if attr_del in node.attributes.keys():
                    node.removeAttribute(attr_del)
            for attr_del, value_del in self.xml_tags_to_delete['remove-attribute-value'].get(node.nodeName, dict()).items():
                if attr_del in node.attributes.keys() and node.attributes[attr_del].value == value_del:
                    node.removeAttribute(attr_del)
        for child in node.childNodes[:]:
            if child.nodeName in self.xml_tags_to_delete['tags'] or child.nodeType == child.COMMENT_NODE or (child.nodeType == child.TEXT_NODE and child.nodeValue.isspace()):
                node.childNodes.remove(child)
            elif child.nodeType in (child.TEXT_NODE, child.CDATA_SECTION_NODE):
                child.nodeValue = child.nodeValue.strip()
                parsed, is_json = self._try_to_parse_json(child.nodeValue)
                if is_json:
                    self.json_painter.apply_json_paints(parsed)
                elif like_html:
                    parsed = self._try_get_json_inside(child.nodeValue)
                child.nodeValue = self._json2str(parsed)
            else:
                if self._preprocess_xml(child, like_html):
                    node.childNodes.remove(child)
        return False

    def _try_parse_xml(self, body, like_html):
        xml = None
        try:
            xml = md.parseString(body)
        except:
            if like_html:
                cut_body = self.remove_single_xml_tags(body, self.html_tags_remove_to_convert_xml)
                try:
                    xml = md.parseString(cut_body)
                except:
                    return body, False
            return body, False
        if xml:
            self._preprocess_xml(xml, like_html)
            return xml, True

    @staticmethod
    def _extract_ssr(entity):
        ssr = entity.get('direct', {}).get('ssr', {})
        if not ssr:
            ssr = entity.get('settings', {}).get('ssr', {})
        return ssr

    def _css_diff(self, css_pre, css_test):
        try:
            css_pre = self.apply_substituttes(css_pre, self.css_substitutes)
            css_test = self.apply_substituttes(css_test, self.css_substitutes)
            return self._string_lists_diff(str(css_pre).split('\n'), str(css_test).split('\n'))
        except:
            return 'Failed to compare CSS'

    def bodies_diff(self, body_pre, body_test):
        def preprocess(body, is_gzip):
            if is_gzip:
                body = self._decompress(body)
            body, is_base64 = self.base64decode(body)
            body = self.apply_substituttes(body, self.body_substitutes)
            body = self._replace_all_base64(body, self.base64_prefixes)

            body, is_json = self._try_to_parse_json(body)
            body, is_xml = self._try_parse_xml(body, False)
            if not is_json and not is_xml and body.endswith("')"):
                s = body[:-2].split("('", 1)
                if len(s) == 2:
                    s[1], is_json = self._try_to_parse_json(s[1])
                    if is_json:
                        padding, body = s
                        body[self.padding_name] = padding

            return body, is_json, is_xml, is_base64

        def extract(data, name):
            result = data.get(name, '')
            try:
                result = base64.urlsafe_b64decode(str(result))
            except TypeError:
                pass
            return result

        def extract_ssr_fields(data, is_json):
            if is_json:
                ssr = self._extract_ssr(data)
                css = extract(ssr, 'css')
                html = extract(ssr, 'html')
                return css, html
            else:
                return '', ''

        entity_pre, is_pre_json, is_pre_xml, pre_base64 = preprocess(body_pre, self.is_pre_gzip)
        entity_test, is_test_json, is_test_xml, test_base64 = preprocess(body_test, self.is_test_gzip)

        css_pre, html_pre = extract_ssr_fields(entity_pre, is_pre_json)
        css_test, html_test = extract_ssr_fields(entity_test, is_test_json)

        entities_diff = self._entities_diff(entity_pre, entity_test, is_pre_json, is_test_json, is_pre_xml, is_test_xml)
        css_diff = self._css_diff(css_pre, css_test)
        html_diff = self._html_diff(html_pre, html_test, self.html_substitutes, self.html_differ_output_path, self.working_dir, self.html_differ_path, self.window_size)

        return entities_diff, css_diff, html_diff, pre_base64 != test_base64

    def diff_response_data(self, response_pre, response_test):
        status_pre, headers_pre, body_pre = self._split_response(response_pre)
        status_test, headers_test, body_test = self._split_response(response_test)

        if self.skip_by_headers_filters(headers_pre, self.filter_by_pre_headers) or self.skip_by_headers_filters(headers_test, self.filter_by_test_headers):
            return self.no_diff_response

        status_diff = self._status_diff(status_pre, status_test)
        headers_diff = self._headers_diff(headers_pre, headers_test)

        entities_diff, css_diff, html_diff, base64_diff = self.bodies_diff(body_pre, body_test)
        has_diff = len(status_diff) + len(headers_diff) + len(entities_diff) + len(css_diff) + len(html_diff) > 0

        return {
            'HasDiff': has_diff,
            'StatusDiff': status_diff,
            'HeadersDiff': headers_diff,
            'EntityBase64': base64_diff,
            'EntitiesDiff': entities_diff,
            'CssDiff': css_diff,
            'HtmlDiff': html_diff,
        }
