from sandbox import sdk2
import sandbox.common.types.resource as ctr
from sandbox.sandboxsdk import environments

from sandbox.projects.resource_types import CACHE_DAEMON
from sandbox.projects.common.dolbilka.resources import DPLANNER_EXECUTABLE

from sandbox.projects.yql.RunYQL2 import RunYQL2
from sandbox.projects.yabs.qa.tasks.BuildAmmoAndStubsFromYT import BuildAmmoAndStubsFromYT, GenAmmoAndStubFromYtBinary
from sandbox.projects.common.yabs.server.util.general import check_tasks, try_get_from_vault

import queries


class BsSsrGetYtRequestData(sdk2.Task):
    class Requirements(sdk2.Task.Requirements):
        environments = (environments.PipEnvironment('yandex-yt'),)

    class Parameters(sdk2.Parameters):
        # common parameters
        description = 'Generate ssr ammo and stubs from bs shoot response dump'
        max_restarts = 3
        kill_timeout = 60 * 60 * 4

        gen_ammo_and_stub_from_yt_binary = sdk2.parameters.LastReleasedResource(
            'Stub and ammo generator binary',
            resource_type=GenAmmoAndStubFromYtBinary,
            state=(ctr.State.READY, ctr.State.NOT_READY),
            required=True,
        )

        cache_daemon_resource = sdk2.parameters.LastReleasedResource(
            'Cache daemon binary',
            resource_type=CACHE_DAEMON,
            state=(ctr.State.READY, ctr.State.NOT_READY),
            required=True,
        )

        d_planner_resource = sdk2.parameters.LastReleasedResource(
            'Dolbilka planner binary',
            resource_type=DPLANNER_EXECUTABLE,
            state=(ctr.State.READY, ctr.State.NOT_READY),
            required=True,
        )

        work_prefix = sdk2.parameters.String(
            'Working directory', default='//home/yabs-cs-sandbox/gen_ssr_stub'
        )

        shoot_task = sdk2.parameters.Task('BS Shoot task')
        testenv_switch_trigger = sdk2.parameters.Bool('Switch Testenv to generated resources', default=False)
        lower_limits = sdk2.parameters.Dict('Lower limits on request count', default={'meta': 1000})
        custom_headers = sdk2.parameters.Dict('Custom headers to be added to stub responses')

        yt_token_vault_name = sdk2.parameters.String(
            'Vault name to get YT token from',
            default='yabs-cs-sb-yt-token',
        )

        yql_token_vault_name = sdk2.parameters.String(
            'Vault name to get YQL token from',
            default='yabs-cs-sb-yql-token',
        )

        with sdk2.parameters.Output:
            result_spec = sdk2.parameters.JSON('Output generated ammo spec', default={})

    class Context(sdk2.Context):
        genammo_subtask_id = None
        resource_generation_subtask_id = None

    def on_execute(self):
        yt_work_prefix = self.Parameters.work_prefix + '/{}'.format(self.id)
        yt_response_dump_path = self.Parameters.shoot_task.Parameters.uploaded_logs_to_yt_prefix

        if self.Context.genammo_subtask_id is None:
            query = queries.make_ssr_ammo_query(
                work_prefix=yt_work_prefix,
                response_dump_prefix=yt_response_dump_path,
                handlers=[str(k) for k in self.Parameters.lower_limits.keys()],
                custom_headers=[(str(k), str(v)) for k, v in self.Parameters.custom_headers.iteritems()],
            )
            self.Context.genammo_subtask_id = self.launch_run_yql2(query, 15, 'genssrammo')

        if self.Context.resource_generation_subtask_id is None:
            check_tasks(self, [self.Context.genammo_subtask_id])

            SPEC_KEY = 'ssr'
            spec = {
                SPEC_KEY: {
                    'make_load': False,
                    'limits': [
                        {'lower': {k: int(v) for k, v in self.Parameters.lower_limits.iteritems()}},
                    ],
                },
            }

            input_tables = {
                SPEC_KEY: [
                    yt_work_prefix + '/output',
                ]
            }
            self.set_tables_attrs(input_tables)
            resource_generation_subtask = BuildAmmoAndStubsFromYT(
                self,
                description='Ssr ammo and stub generation',
                gen_ammo_and_stub_from_yt_binary=self.Parameters.gen_ammo_and_stub_from_yt_binary,
                cache_daemon_resource=self.Parameters.cache_daemon_resource,
                d_planner_resource=self.Parameters.d_planner_resource,
                input_tables=input_tables,
                spec=spec,
                yt_token_vault_name=self.Parameters.yt_token_vault_name,
                yt_work_prefix=yt_work_prefix,
                legacy_mode=False,
                testenv_switch_trigger=self.Parameters.testenv_switch_trigger,
            )
            resource_generation_subtask.enqueue()
            self.Context.resource_generation_subtask_id = resource_generation_subtask.id

        check_tasks(self, [self.Context.resource_generation_subtask_id])

        result_spec = sdk2.Task[self.Context.resource_generation_subtask_id].Parameters.result_spec
        self.Parameters.result_spec = result_spec

    def launch_run_yql2(self, query, retry_period, subdescr):
        subtask = RunYQL2(
            self,
            description='Ssr ammo and stub generation: {}'.format(subdescr),
            query=query,
            yql_token_vault_name=self.Parameters.yql_token_vault_name,
            trace_query=True,
            retry_period=retry_period,
            publish_query=True,
            use_v1_syntax=True,
        )
        subtask.enqueue()
        return subtask.id

    def set_tables_attrs(self, input_tables):
        import yt.wrapper as yt

        yt.config['token'] = try_get_from_vault(self, self.Parameters.yt_token_vault_name)
        yt.config['proxy']['url'] = 'hahn'
        for table_list in input_tables.itervalues():
            for table_path in table_list:
                yt.set(table_path + '/@request_key_headers', ['x-yabs-ssr-req-id'])
                yt.set(table_path + '/@cachedaemon_key_headers', ['x-yabs-ssr-req-id'])
