import os
import subprocess

from sandbox import sdk2
from sandbox.sdk2.vcs import svn
from sandbox.common.types.misc import DnsType
from sandbox.projects.common.yabs.server.util.general import try_get_from_vault
from sandbox.projects.yabs.ssr.resources import BsSsrPcodeResource
from sandbox.projects.yabs.ssr.util import run_command


NVM_URL = 'https://raw.githubusercontent.com/nvm-sh/nvm/v0.37.2/install.sh'
CONTAINER_RESOURCE = 2090878873
LOCAL_ARCADIA_DIR = 'arcadia'
NVMRC_PATH = 'adv/pcode/web/pcode/.nvmrc'


class BsSsrPrepareRendererTask(sdk2.Task):
    class Requirements(sdk2.Requirements):
        container_resource = CONTAINER_RESOURCE
        dns = DnsType.DNS64

    class Parameters(sdk2.Task.Parameters):
        description = 'Task for PCODE renderer preparation.'
        working_dir = sdk2.parameters.String(
            'Name for directory with code',
            default='renderer',
        )
        repo_commit = sdk2.parameters.String('Commit branch will be reset on')
        ya_token_vault_name = sdk2.parameters.String(
            "YA token vault name",
            default='ssr_tests_ya_token',
        )
        user_email = sdk2.parameters.String(
            'User email used for authentification',
            default='geras-artem@yandex-team.ru',
        )

        with sdk2.parameters.Output:
            renderer_resource = sdk2.parameters.Resource('Resource with BS SSR data', resource_type=BsSsrPcodeResource)

    def checkout(self, path):
        arcadia_client = svn.Arcadia
        checkout_dir = os.path.join(arcadia_client.ARCADIA_TRUNK_URL, os.path.dirname(path))
        local_dir = os.path.join(LOCAL_ARCADIA_DIR, os.path.dirname(path))
        if not os.path.exists(local_dir):
            arcadia_client.checkout(url=checkout_dir, path=local_dir)
        path = os.path.join(LOCAL_ARCADIA_DIR, path)
        assert os.path.exists(path)
        return path

    def _load_code(self):
        def run_command_with_stderr(command):
            return run_command(command, stderr=subprocess.STDOUT)

        root_dir = os.getcwd()
        working_dir = self.Parameters.working_dir
        os.makedirs(working_dir)
        os.chdir(working_dir)

        working_dir = os.getcwd()
        os.makedirs('.nvm')
        os.environ['NVM_DIR'] = os.path.join(working_dir, '.nvm')

        nvmrc_path = self.checkout(NVMRC_PATH)
        with open(nvmrc_path, 'r') as f:
            version = f.readline().strip(' \n')

        os.chdir(os.path.dirname(nvmrc_path))
        run_command_with_stderr(
            'curl -o- "{nvm_url}" | bash && '
            'source "$NVM_DIR/nvm.sh" || true && '
            'nvm install {version}'.format(
                nvm_url=NVM_URL,
                version=version,
            )
        )

        node_path = os.path.join(os.environ['NVM_DIR'], 'versions', 'node', 'v' + version, 'bin')
        os.environ['PATH'] += os.pathsep + node_path

        run_command_with_stderr('npm config set registry https://npm.yandex-team.ru')
        run_command_with_stderr(
            'npm config set email "{user_email}"'.format(
                user_email=self.Parameters.user_email,
            )
        )
        run_command_with_stderr(
            'npm config set _auth {token}'.format(
                token=try_get_from_vault(self, self.Parameters.ya_token_vault_name),
            )
        )

        run_command_with_stderr('npm install yarn')
        run_command_with_stderr('npm install --only=dev @yandex-int/archon @yandex-int/archon-renderer')
        run_command_with_stderr('npx yarn || true')

        run_command_with_stderr('npm install --only=dev direct-ad-adapter')

        os.chdir(root_dir)

        return os.path.relpath(node_path, working_dir)

    def on_execute(self):
        renderer_resource = BsSsrPcodeResource(
            self,
            'BS SSR resource',
            self.Parameters.working_dir,
            repo_commit=self.Parameters.repo_commit,
        )
        renderer_resource.node_relpath = self._load_code()
        self.Parameters.renderer_resource = renderer_resource
