import json
import logging
import os
import subprocess

from sandbox import sdk2
from sandbox.sandboxsdk import environments

from sandbox.projects.common.yabs.server.util.general import try_get_from_vault
from sandbox.projects.yabs.ssr.resources import BsSsrPcodeResource, AdaptedMordaResponses
from sandbox.sandboxsdk.channel import channel

CHUNK_SIZE = 128
CONTAINER_RESOURCE = 2090878873


class UniformatMordaAdapter(sdk2.Task):
    class Requirements(sdk2.Task.Requirements):
        container_resource = CONTAINER_RESOURCE
        environments = (
            environments.PipEnvironment('yandex-yt'),
            environments.PipEnvironment('yandex-yt-yson-bindings'),
            environments.PipEnvironment('yandex-yt-yson-bindings-skynet'),
        )

    class Parameters(sdk2.Parameters):
        # common parameters
        description = 'Generate ssr ammo and stubs from bs shoot response dump'
        max_restarts = 3
        kill_timeout = 60 * 60 * 4

        with sdk2.parameters.Group('YT settings') as yt_settings:
            yt_token = sdk2.parameters.String('YT token name in Sandbox vault', default='yabs-cs-sb-yt-token')

        shoot_task = sdk2.parameters.Task('BS Shoot task')
        adapter_chunk_size = sdk2.parameters.Integer('Number of responses processed simultaneously', default=100000)

        pcode_package_resource = sdk2.parameters.Resource(
            'Resource with PCODE content',
            resource_type=BsSsrPcodeResource,
        )

        with sdk2.parameters.Output:
            adapted_responses_resource_id = sdk2.parameters.Resource('ResourceID with adapted responses', resource_type=AdaptedMordaResponses)

    @staticmethod
    def read_yt_table(yt, table_path, key_column, columns=None):
        columns_to_read = [key_column] + columns if columns is not None else None
        result = {}
        for row in yt.read_table(yt.TablePath(table_path, columns=columns_to_read)):
            key_value = row[key_column]
            columns_value = {col: row[col] for col in columns} if columns is not None else row
            result[key_value] = columns_value
        return result

    @staticmethod
    def sync_resource(id):
        return channel.task.sync_resource(id)

    def _unpack_renderer(self):
        pcode_resource_path = self.sync_resource(self.Parameters.pcode_package_resource)
        work_dir = os.getcwd()
        self._pcode_unpacked_path = os.path.join(work_dir, 'renderer')

        logging.info('Unpacking pcode resource to {}'.format(self._pcode_unpacked_path))
        with sdk2.helpers.ProcessLog(self, 'adapter.unpack') as process_log:
            subprocess.Popen(
                ['tar', '-xvf', pcode_resource_path, '-C', work_dir],
                stdout=process_log.stdout,
                stderr=process_log.stderr,
            ).wait()
        os.environ['PATH'] += os.pathsep + os.path.join(self._pcode_unpacked_path,
                                                        self.Parameters.pcode_package_resource.node_relpath)

    def _run_adapter(self, files):
        root_dir = os.getcwd()
        os.chdir(self.responses_dir)

        cmd = [
            'npx',
            '--no-install',
            'direct-ad-adapter',
        ]
        cmd.extend(files)
        logging.info('Running adapter with command: {}'.format(cmd))

        self._renderer_process_log_context = sdk2.helpers.ProcessLog(self, 'adapter.run')
        self._renderer_process_log_context.__enter__()
        self._renderer_process = subprocess.Popen(
            cmd, stdout=self._renderer_process_log_context.stdout, stderr=self._renderer_process_log_context.stderr
        )
        self._renderer_process.wait()
        os.chdir(root_dir)

    def _dump_single_response(self, request_id, response):
        file_name = '{}.json'.format(request_id)
        file_path = os.path.join(self.responses_dir, file_name)
        out_file_name = os.path.join(self.responses_dir, '{}.out.json'.format(request_id))
        with open(file_path, 'w') as f:
            f.write(response)
        return file_name, out_file_name

    def _load_single_response(self, file_name, out_file_name):
        with open(out_file_name, 'r') as f:
            adapted_response = f.read()
        os.remove(os.path.join(self.responses_dir, file_name))
        os.remove(out_file_name)
        return adapted_response

    def _dump_responses_wrap(self, args):
        try:
            return self.dump_responses(*args)
        except:
            logging.exception('Failed to process results')
            raise

    def process_files(self):
        self._run_adapter(self.files_to_dump)
        adapted_responses = []
        for f_in, f_out in zip(self.files_to_dump, self.files_to_parse):
            adapted_responses.append(self._load_single_response(f_in, f_out))
        self.files_to_dump = []
        self.files_to_parse = []
        return adapted_responses

    def dump_responses(self, request_id, response):
        file, out_file = self._dump_single_response(request_id, response)
        self.files_to_dump.append(file)
        self.files_to_parse.append(out_file)
        if len(self.files_to_dump) >= self.Parameters.adapter_chunk_size:
            return self.process_files()
        return None

    def get_result(self, data, results):
        results = [r for r in results if r is not None]
        flat_res = []
        for res in results:
            flat_res.extend(res) if type(res) == list else flat_res.append(res)

        result_dict = {
            request_id: {
                'Data': '{status}\r\n{headers}\r\n\r\n{body}'.format(status=data[request_id]['HttpCode'],
                                                                     headers=data[request_id]['Headers'], body=body),
                'Url': 'meta/morda',
                'HttpCode': data[request_id]['HttpCode'],
            }
            for request_id, body in dict(zip(data.keys(), flat_res)).items()
        }
        return result_dict

    def on_execute(self):
        import yt.wrapper as yt

        yt.config['proxy']['url'] = 'hahn'
        yt.config['token'] = try_get_from_vault(self, self.Parameters.yt_token)

        def read_data(yt_path):
            logging.info('Reading shoot results from path: {}'.format(yt_path))
            return self.read_yt_table(yt, yt_path, 'RequestID', ['Response', 'Headers', 'HttpCode'])

        self._unpack_renderer()
        self.files_to_dump = []
        self.files_to_parse = []
        self.responses_dir = os.path.join(self._pcode_unpacked_path, 'repo')

        yt_response_dump_path = self.Parameters.shoot_task.Parameters.uploaded_logs_to_yt_prefix + '/primary_ammo/0/response'
        data = read_data(yt_response_dump_path)
        logging.info('Data collected')

        dump_args = (
            (request_id, data[request_id]['Response']) for request_id in data.keys()
        )

        results = map(self._dump_responses_wrap, dump_args)
        if self.files_to_parse:
            logging.info('Running adapter for tail')
            results.append(self.process_files())

        result_dict = self.get_result(data, results)
        out_resource = AdaptedMordaResponses(self, 'Adapted responses', 'adapted_responses.json')
        with open(str(out_resource.path), 'w') as f:
            f.write(json.dumps(result_dict))
        self.Parameters.adapted_responses_resource_id = out_resource
