from __future__ import print_function

import json
import logging
import time

from sandbox.sdk2 import (
    Vault,
    parameters,
    Requirements,
)
from sandbox.projects.yabs.base_bin_task import BaseBinTask

# https://wiki.yandex-team.ru/yt/userdoc/dynamictablesmapreduce/#prevrashheniestaticheskojjtablicyvdinamicheskuju
COPY_SPEC = {"job_io": {"table_writer": {"block_size": 256 * 2**10, "desired_chunk_size": 100 * 2**20}}}


class YabsStaticToDynamicCopier(BaseBinTask):
    class Requirements(Requirements):
        cores = 1
        ram = 2048
        disk_space = 2048

        class Caches(Requirements.Caches):
            pass

    class Parameters(BaseBinTask.Parameters):
        with BaseBinTask.Parameters.version_and_task_resource() as version_and_task_resource:
            resource_attrs = parameters.Dict("Filter resource by", default={"name": "YabsStaticToDynamic"})

        with parameters.Group('Copying parameters') as copy_params:
            yt_token_vault = parameters.String('Vault with yt token', required=True)
            cluster = parameters.String('YT Cluster', required=True)
            meta_cluster = parameters.String('YT cluster with meta info', required=True)
            meta_path = parameters.String('Meta path', required=True)
            local_meta_path = parameters.String('Local meta path', required=True)
            dst_path = parameters.String('Destination work dir', required=True)
            link_path = parameters.String('Link path', required=True)
            tablet_count = parameters.Integer('Tablet count', default=50, required=True)
            user_attrs = parameters.Dict('Additional table attributes', default={})
            force_compaction = parameters.Bool('Force compaction')

        with parameters.Group('Transfer manager') as tm_params:
            tm_queue = parameters.String(
                'TransferManager queue',
            )
            tm_pool = parameters.String(
                'TransferManager YT pool',
            )

    def get_metadata(self):
        from yt.wrapper import YtClient

        meta_ytc = YtClient(proxy=self.Parameters.meta_cluster, token=self.yt_token)
        global_meta = {}
        if meta_ytc.exists(self.Parameters.meta_path):
            global_meta = json.loads(meta_ytc.read_file(self.Parameters.meta_path).read())

        local_meta = {}
        if self.ytc.exists(self.Parameters.local_meta_path):
            local_meta = json.loads(self.ytc.read_file(self.Parameters.local_meta_path).read())

        logging.info('Meta: global: %s, local: %s', global_meta, local_meta)
        return global_meta, local_meta

    def wait_table(self, table, state):
        logging.info('Waiting for table %s to enter state %s', table, state)
        while not all(x["state"] == state for x in self.ytc.get_attribute(table, "tablets")):
            time.sleep(1)

    def do_copy(self, global_meta):
        from yt.transfer_manager.client import TransferManager

        table = '{}/{}'.format(self.Parameters.dst_path, global_meta['timestamp'])
        self.ytc.remove(table, force=True)

        params = {'copy_spec': COPY_SPEC}
        if self.Parameters.tm_pool:
            pool = self.Parameters.tm_pool
            params['copy_spec']['pool'] = pool
            params['postprocess_spec'] = {'pool': pool}
        if self.Parameters.tm_queue:
            params['queue_name'] = self.Parameters.tm_queue

        tmc = TransferManager(token=self.yt_token)
        logging.info('Transfer: %s.[%s] -> %s.[%s]', global_meta['cluster'], global_meta['path'], self.Parameters.cluster, table)
        tmc.add_task(
            global_meta['cluster'],
            global_meta['path'],
            self.Parameters.cluster,
            table,
            params=params,
            sync=True
        )

        self.ytc.alter_table(table, dynamic=True)
        self.ytc.reshard_table(table, tablet_count=self.Parameters.tablet_count)
        self.wait_table(table, 'unmounted')

        for attr, val in self.Parameters.user_attrs.iteritems():
            self.ytc.set_attribute(table, attr, val)

        self.ytc.mount_table(table)
        self.wait_table(table, 'mounted')

        if self.Parameters.force_compaction:
            logging.info("Force compaction table %s", table)
            self.ytc.set_attribute(table, "forced_compaction_revision", self.ytc.get_attribute(table, "revision"))
            self.ytc.set_attribute(table, "forced_compaction_revision", self.ytc.get_attribute(table, "revision"))
            self.ytc.remount_table(table)

        logging.info('Table %s is ready', table)
        return table

    def update_link(self, table, new_meta):
        prev_table = None
        if self.ytc.exists(self.Parameters.link_path):
            prev_table = self.ytc.get_attribute(self.Parameters.link_path + '&', 'target_path', None)

        with self.ytc.Transaction():
            logging.info('Link %s -> %s', self.Parameters.link_path, table)
            self.ytc.link(table, self.Parameters.link_path, force=True)
            self.ytc.write_file(self.Parameters.local_meta_path, json.dumps(new_meta))
            for path in self.ytc.list(self.Parameters.dst_path, absolute=True):
                if path not in (prev_table, table, self.Parameters.local_meta_path):
                    logging.info('Removing old table %s', path)
                    self.ytc.remove(path)

    def on_execute(self):
        from yt.wrapper import YtClient

        yt_token = Vault.data(self.Parameters.yt_token_vault)
        self.yt_token = yt_token
        self.ytc = YtClient(proxy=self.Parameters.cluster, token=self.yt_token)

        global_meta, local_meta = self.get_metadata()
        if global_meta.get('timestamp', '') <= local_meta.get('timestamp', ''):
            logging.info('No new table found, stopping')
            return

        new_table = self.do_copy(global_meta)
        self.update_link(new_table, global_meta)
