import time

from sandbox.sdk2 import (
    Requirements,
    parameters,
)
from sandbox.projects.yabs.base_bin_task import BaseBinTask


YQL_QUERY_PRECENTILES = '''
pragma yt.QueryCacheMode('disable');

FROM (
    FROM `{path}`
    SELECT
        MAX_OF(last_update_lag, CAST(CurrentUtcTimestamp() AS int64) / 1000000 - updated) as last_update_lag,
        elapsed_time
    WHERE
        status != 4
)
SELECT
    PERCENTILE(last_update_lag, 0.50) as lag50,
    PERCENTILE(last_update_lag, 0.75) as lag75,
    PERCENTILE(last_update_lag, 0.95) as lag95,
    PERCENTILE(last_update_lag, 0.99) as lag99,
    MAX(last_update_lag) as lag100,
    PERCENTILE(elapsed_time, 0.50) as time50,
    PERCENTILE(elapsed_time, 0.75) as time75,
    PERCENTILE(elapsed_time, 0.95) as time95,
    PERCENTILE(elapsed_time, 0.99) as time99
'''

PRECENTAGES = (50, 75, 95, 99)

YQL_QUERY_STATUSES = '''
pragma yt.QueryCacheMode('disable');

FROM `{path}`
SELECT
    status,
    COUNT(*) as cnt
GROUP BY
    status
'''


class YabsUrlMonitoringStatistics(BaseBinTask):
    """Push states table statistics to solomon"""
    class Requirements(Requirements):
        cores = 1

        class Caches(Requirements.Caches):
            pass

    class Parameters(BaseBinTask.Parameters):
        max_restarts = 1

        with BaseBinTask.Parameters.version_and_task_resource() as version_and_task_resource:
            resource_attrs = parameters.Dict("Filter resource by", default={"name": "YabsUrlMonitoring"})

        with parameters.Group("YT parameters") as yt_parameters:
            with parameters.String("Cluster", default="hahn") as cluster:
                cluster.values.hahn = "hahn"
                cluster.values.banach = "banach"
                cluster.values.freud = "freud"

            yav_secret = parameters.YavSecret(
                label="Yav secret id",
                required=True,
                description='secret should contain keys: yabs-url-mon-yt-token, yabs-url-mon-solomon-token',
                default="sec-01ct3gky5hfxyjer8z00qaw31v",
            )

        with parameters.Group("Input parameters") as in_parameters:
            states_table = parameters.String("States table", default="//home/yabs/url_monitoring/states")

        with parameters.Group("Statistics parameters") as statistics:
            host = parameters.String("Host to label metrics", default="sandbox")
            statistics_cluster = parameters.String("Solomon cluster", default="preprod")
            project = parameters.String("Solomon project", default="yabs_url_monitoring")
            solomon_api_url = parameters.String("Solomon API URL", default="https://solomon.yandex.net/api/v2/push")

    def generate_config(self):
        from yabs.stat.url_monitoring2.lib.config import Config

        d = {
            "statistics": {
                "host": self.Parameters.host,
                "project": self.Parameters.project,
                "cluster": self.Parameters.statistics_cluster,
                "solomon_api_url": self.Parameters.solomon_api_url,
            }
        }

        config = Config(d)
        return config

    def do_calculate_statistics(self, yql_client, config, states_table, solomon_token):
        from yabs.stat.url_monitoring2.lib.statistics import Statistics

        now = int(time.time())
        stats = Statistics(solomon_token)

        request = yql_client.query(YQL_QUERY_PRECENTILES.format(path=states_table), syntax_version=1)
        request.run()
        request_url = request.share_url
        self.set_info("Precentiles YQL query: {}".format(request_url))

        results = request.get_results()
        table = None
        for res in results:
            table = res
            break

        result = next(table.get_iterator())
        result = dict(zip(table.column_names, result))

        for prec in (50, 75, 95, 99):
            stats.add(service="total", labels={"type": "elapsed_time", "precentage": prec}, ts=now, value=result["time{}".format(prec)])
            stats.add(service="total", labels={"type": "last_update_lag", "precentage": prec}, ts=now, value=result["lag{}".format(prec)])
        stats.add(service="total", labels={"type": "last_update_lag", "precentage": 100}, ts=now, value=result["lag100"])

        request = yql_client.query(YQL_QUERY_STATUSES.format(path=states_table), syntax_version=1)
        request.run()
        request_url = request.share_url
        self.set_info("Statuses YQL query: {}".format(request_url))

        for table in request.get_results():
            for row in table.get_iterator():
                stats.add(service="total", labels={"type": "status", "status": row[0]}, ts=now, value=row[1])

        stats.send_sensors_to_solomon(config)

    def on_execute(self):
        from yql.api.v1.client import YqlClient

        yql_token = self.Parameters.yav_secret.data()["yabs-url-mon-yt-token"]
        solomon_token = self.Parameters.yav_secret.data()["yabs-url-mon-solomon-token"]

        config = self.generate_config()
        self.do_calculate_statistics(YqlClient(token=yql_token, db=self.Parameters.cluster), config, self.Parameters.states_table, solomon_token)
