import os
import logging
from sandbox.common.types import task as ctt
from sandbox.common.types import notification as ctn
from sandbox.projects.common import binary_task
from sandbox import sdk2

logger = logging.getLogger()


class FromSandboxToYt(binary_task.LastBinaryTaskRelease, sdk2.Task):

    class Parameters(sdk2.Task.Parameters):
        description = "Storing to Yt normalization data"
        notifications = [
            sdk2.Notification(
                statuses=[
                    ctt.Status.FAILURE,
                    ctt.Status.EXCEPTION,
                    ctt.Status.TIMEOUT
                ],
                recipients=["pogodina-eval@yandex-team.ru"],
                transport=ctn.Transport.EMAIL
            )
        ]
        mirrors_resource_type = sdk2.parameters.String(
            "Mirrors resource type", required=True, default="CONTEXT_MASTER_MIRRORS"
        )
        mirrors_resource_path = sdk2.parameters.String(
            "Mirrors resource path", required=True, default="mirrors.map"
        )
        robots_resource_type = sdk2.parameters.String(
            "Robots resource type", required=True, default="CONTEXT_ROBOTS_CLEAN_PARAMS"
        )
        robots_resource_path = sdk2.parameters.String(
            "Robots resource path", required=True, default="robots_clean_params.dump"
        )
        mirrors_table = sdk2.parameters.String(
            "Path to store mirrors data",
            default="//home/bs/users/pogodina_eval/upload/mirrors",
            required=True
        )
        robots_table = sdk2.parameters.String(
            "Path to store robots data",
            default="//home/bs/users/pogodina_eval/upload/robots",
            required=True
        )
        yt_token_secret = sdk2.parameters.YavSecret(
            "Secret to token",
            required=True,
            default="sec-01g4cwncb5xx7m1vn8cswmr8y7"
        )
        ext_params = binary_task.binary_release_parameters(stable=True)

    def on_execute(self):
        import yt.wrapper as yt
        from blob.blob import mirrors_dump, robots_dump

        yt.config['token'] = self.Parameters.yt_token_secret.data().get("yt_token")

        mirrors_resource = sdk2.Resource.find(
            type=self.Parameters.mirrors_resource_type,
            state="READY",
            attrs={"released": "stable"}
        ).first()
        robots_resource = sdk2.Resource.find(
            type=self.Parameters.robots_resource_type,
            state="READY",
            attrs={"released": "stable"}
        ).first()

        mirrors_file_path = os.path.join(str(sdk2.ResourceData(mirrors_resource).path), self.Parameters.mirrors_resource_path)
        robots_file_path = os.path.join(str(sdk2.ResourceData(robots_resource).path), self.Parameters.robots_resource_path)
        mirrors_dump_string = mirrors_dump(mirrors_file_path)
        robots_dump_string = robots_dump(robots_file_path)
        logger.info("LEN STRING {}".format(len(robots_dump_string)))

        ROBOTS_BATCH_SIZE = 10000000
        ROBOTS_DUMP_SIZE = len(robots_dump_string)
        for cluster in ["hahn", "arnold"]:
            yt.config.set_proxy(cluster)
            yt.write_table(self.Parameters.mirrors_table, [{"blob": mirrors_dump_string}])
            yt.write_table(
                self.Parameters.robots_table,
                [{"id": batch, "blob": robots_dump_string[ROBOTS_BATCH_SIZE*batch:ROBOTS_BATCH_SIZE*(batch+1)]}
                 for batch in range(ROBOTS_DUMP_SIZE // ROBOTS_BATCH_SIZE + 1)]
            )
