# -*- coding: utf-8 -*-
import re
from sandbox import sdk2
from sandbox.projects import rope
from datetime import datetime, timedelta

YABS_ANALYTICS_TASK_BINARIES_RESOURCES = []
YABS_ANALYTICS_TASK_BINARIES_RESOURCES_RUNNER = []
YABS_ANALYTICS_TASK_NAMES = set()

YABS_ANALYTICS_RELEASERS = ['rusnarziev']
GENERAL_TTL = 30


def default_decorator(decorator):
    def wrapper(*args, **kwargs):
        if not kwargs and len(args) == 1:
            fn, = args
            return decorator()(fn)
        return decorator(*args, **kwargs)

    return wrapper


@default_decorator
def register_as_yabs_analytics_task(custom_runner=False):
    """
    :param bool custom_runner: if you want to implement custom runner
        this task will be excluded from common task runner
    """
    def decorator(cls):
        task_name = rope.RunTaskTemplate.get_resource_task_name(cls)
        if not re.match(r'^[a-z][a-z0-9]*(_[a-z0-9]+)*$', task_name):
            raise ValueError('Wrong task name "{}" - should be in snake_case.'.format(task_name))
        if task_name in YABS_ANALYTICS_TASK_NAMES:
            raise ValueError('Task with name {} already exists'.format(task_name))
        YABS_ANALYTICS_TASK_BINARIES_RESOURCES.append(cls)
        if not custom_runner:
            YABS_ANALYTICS_TASK_BINARIES_RESOURCES_RUNNER.append(
                (cls, getattr(cls, 'TaskParams', None)))
        YABS_ANALYTICS_TASK_NAMES.add(task_name)
        cls.__is_yabs_analytics_task__ = True
        return cls
    return decorator


class YabsAnalyticsBaseTaskParams(rope.parameters.TaskParams):
    yt_yql_cluster = rope.parameters.StrParam(
        'YT/YQL cluster',
        default='hahn',
        required=True,
        sdk2_param_class=sdk2.parameters.RadioGroup,
        sdk2_param_kwargs=dict(
            choices=(
                ('hahn', 'hahn'),
                ('arnold', 'arnold'),
            )
        ),
    )
    yt_proxy = rope.parameters.StrParam(
        'YT cluster',
        default=lambda p: p.yt_yql_cluster,
        skip_in=rope.parameters.ParamSrc.SANDBOX_SDK2_PARAM)
    yql_proxy = rope.parameters.StrParam(
        'YQL cluster',
        default=lambda p: p.yt_yql_cluster,
        skip_in=rope.parameters.ParamSrc.SANDBOX_SDK2_PARAM)

    yt_token = rope.parameters.VaultSecretParam(
        'YT token',
        default_vault_name='robot-hobot_yt_token',
        default_vault_user='YABS_ANALYTICS',
        required=True,
        specify_owner=True)
    yql_token = rope.parameters.VaultSecretParam(
        'YQL token',
        default_vault_name='robot-hobot_yql_token',
        default_vault_user='YABS_ANALYTICS',
        required=True,
        specify_owner=True)

    yt_pool = rope.parameters.StrParam('YT pool')
    yql_pragmas = rope.parameters.DictParam(
        'YQL pragmas',
        default={'OrderedColumns': None})

    solomon_token = rope.parameters.VaultSecretParam(
        'Solomon token',
        default_vault_name='robot-hobot_solomon_token',
        default_vault_user='YABS_ANALYTICS',
        required=True,
        specify_owner=True)
    solomon_project = rope.parameters.StrParam(
        'Solomon project',
        default='yabs-analytics')

    task_type = rope.parameters.StrParam(
        'Task type',
        default='test',
        required=True,
        sdk2_param_class=sdk2.parameters.RadioGroup,
        sdk2_param_kwargs=dict(
            choices=(
                ('prod', 'prod'),
                ('test', 'test'),
                ('debug', 'debug')
            )
        )
    )

    use_solomon_scheduler = rope.parameters.BoolParam('Push schedule metrics in solomon', default=True)
    overwrite_output = rope.parameters.BoolParam('Overwrite output', default=False)
    debug = rope.parameters.BoolParam('Debug mode', default=False)


@register_as_yabs_analytics_task
class Adhoc(sdk2.Resource):
    executable = True
    releasers = YABS_ANALYTICS_RELEASERS
    ttl = GENERAL_TTL

    arcadia_build_path = 'yabs/analytics/adhoc/adhoc'
    task_name = 'adhoc'
    task_executable_resource_description = 'Yabs analytics adhoc executable binaries'

    class TaskParams(YabsAnalyticsBaseTaskParams):
        config_path = rope.parameters.StrParam('Config path',
                                               default='yabs/analytics/adhoc/conf')
        ticket = rope.parameters.StrParam('Ticket', required=True)
        output_folder = rope.parameters.StrParam('Output folder',
                                                 default='//home/bs/users/yabs-analytics/adhoc',
                                                 required=True)
        query_type = rope.parameters.StrParam(
            'Query type',
            required=True,
            sdk2_param_class=sdk2.parameters.RadioGroup,
            sdk2_param_kwargs=dict(
                choices=(
                    ('select', 'select'),
                    ('stat', 'stat'),
                    ('valid', 'valid')
                )
            ), default='select'
        )
        config = rope.parameters.StrParam('Config', required=True)


@register_as_yabs_analytics_task
class AnomalyAnalyzerCollector(sdk2.Resource):
    executable = True
    releasers = YABS_ANALYTICS_RELEASERS
    ttl = GENERAL_TTL

    arcadia_build_path = 'yabs/analytics/anomaly_analyzer/collector/collector'
    task_name = 'anomaly_analyzer_collector'
    task_executable_resource_description = 'Yabs analytics anomaly_analyzer_collector executable binaries'

    class TaskParams(YabsAnalyticsBaseTaskParams):
        config_path = rope.parameters.StrParam('Config path',
                                               default='yabs/analytics/anomaly_analyzer/collector/conf.yaml')
        output_folder = rope.parameters.StrParam('Output folder',
                                                 default='//home/bs/users/yabs-analytics/anomaly_analyzer/collector',
                                                 required=True)
        start_date = rope.parameters.DateParam('Start date',
                                               details='YYYY-MM-DD format (if not set yesterday date will be used)',
                                               default=lambda p: (datetime.now() - timedelta(days=1)).date(),
                                               skip_in=rope.parameters.ParamSrc.SANDBOX_SDK2_PARAM)
        end_date = rope.parameters.DateParam('End date',
                                             details='YYYY-MM-DD format (if not set yesterday date will be used)',
                                             default=lambda p: (datetime.now() - timedelta(days=1)).date(),
                                             skip_in=rope.parameters.ParamSrc.SANDBOX_SDK2_PARAM)


@register_as_yabs_analytics_task
class AutobudgetAllowedSpendingMonitoring(sdk2.Resource):
    executable = True
    releasers = YABS_ANALYTICS_RELEASERS
    ttl = GENERAL_TTL

    arcadia_build_path = 'yabs/analytics/autobudget/allowed_spending_monitoring/allowed_spending_monitoring'
    task_name = 'autobudget_allowed_spending_monitoring'
    task_executable_resource_description = 'Yabs analytics autobudget_allowed_spending_monitoring executable binaries'

    class TaskParams(YabsAnalyticsBaseTaskParams):
        config_path = rope.parameters.StrParam(
            'Config path', default='yabs/analytics/autobudget/allowed_spending_monitoring/conf.yaml')
        output_folder = rope.parameters.StrParam('Output folder',
                                                 default='//home/ads/cpa_autobudget/data/allowed_spending_monitoring',
                                                 required=True)
        start_date = rope.parameters.DateParam('Start date',
                                               details='YYYY-MM-DD format (if not set yesterday date will be used)',
                                               default=lambda p: (datetime.now() - timedelta(days=1)).date())
        end_date = rope.parameters.DateParam('End date',
                                             details='YYYY-MM-DD format (if not set yesterday date will be used)',
                                             default=lambda p: (datetime.now() - timedelta(days=1)).date())


@register_as_yabs_analytics_task
class AutobudgetPaidActionsMonitoring(sdk2.Resource):
    executable = True
    releasers = YABS_ANALYTICS_RELEASERS
    ttl = GENERAL_TTL

    arcadia_build_path = 'yabs/analytics/autobudget/paid_actions_monitoring/paid_actions_monitoring'
    task_name = 'autobudget_paid_actions_monitoring'
    task_executable_resource_description = 'Yabs analytics autobudget_paid_actions_monitoring executable binaries'

    class TaskParams(YabsAnalyticsBaseTaskParams):
        config_path = rope.parameters.StrParam('Config path',
                                               default='yabs/analytics/autobudget/paid_actions_monitoring/conf.yaml')
        output_folder = rope.parameters.StrParam('Output folder',
                                                 default='//home/ads/cpa_autobudget/data/paid_actions_monitoring',
                                                 required=True)
        start_date = rope.parameters.DateParam('Start date',
                                               details='YYYY-MM-DD format (if not set yesterday date will be used)',
                                               default=lambda p: (datetime.now() - timedelta(days=1)).date())
        end_date = rope.parameters.DateParam('End date',
                                             details='YYYY-MM-DD format (if not set yesterday date will be used)',
                                             default=lambda p: (datetime.now() - timedelta(days=1)).date())


@register_as_yabs_analytics_task
class ReachAdsFunnel(sdk2.Resource):
    executable = True
    releasers = YABS_ANALYTICS_RELEASERS
    ttl = GENERAL_TTL

    arcadia_build_path = 'yabs/analytics/reach/ads_funnel/ads_funnel'
    task_name = 'reach_ads_funnel'
    task_executable_resource_description = 'Yabs analytics reach_ads_funnel executable binaries'

    class TaskParams(YabsAnalyticsBaseTaskParams):
        config_path = rope.parameters.StrParam('Config path',
                                               default='yabs/analytics/reach/ads_funnel/conf.yaml')
        output_folder = rope.parameters.StrParam('Output folder',
                                                 default='//home/bs/users/yabs-analytics/reach/ads_funnel',
                                                 required=True)
        start_date = rope.parameters.DateParam('Start date',
                                               details='YYYY-MM-DD format (if not set yesterday date will be used)',
                                               default=lambda p: (datetime.now() - timedelta(days=1)).date())
        end_date = rope.parameters.DateParam('End date',
                                             details='YYYY-MM-DD format (if not set yesterday date will be used)',
                                             default=lambda p: (datetime.now() - timedelta(days=1)).date())


@register_as_yabs_analytics_task
class ReachCpmUnderimpressions(sdk2.Resource):
    executable = True
    releasers = YABS_ANALYTICS_RELEASERS
    ttl = GENERAL_TTL

    arcadia_build_path = 'yabs/analytics/reach/cpm_underimpressions/cpm_underimpressions'
    task_name = 'reach_cpm_underimpressions'
    task_executable_resource_description = 'Yabs analytics reach_cpm_underimpressions executable binaries'

    class TaskParams(YabsAnalyticsBaseTaskParams):
        config_path = rope.parameters.StrParam('Config path',
                                               default='yabs/analytics/reach/cpm_underimpressions/conf.yaml')
        output_folder = rope.parameters.StrParam('Output folder',
                                                 default='//home/bs/users/yabs-analytics/reach/cpm_underimpressions',
                                                 required=True)
        start_date = rope.parameters.DateParam('Start date',
                                               details='YYYY-MM-DD format (if not set yesterday date will be used)',
                                               default=lambda p: (datetime.now() - timedelta(days=1)).date())
        end_date = rope.parameters.DateParam('End date',
                                             details='YYYY-MM-DD format (if not set yesterday date will be used)',
                                             default=lambda p: (datetime.now() - timedelta(days=1)).date())


@register_as_yabs_analytics_task
class ReachDailyDistribution(sdk2.Resource):
    executable = True
    releasers = YABS_ANALYTICS_RELEASERS
    ttl = GENERAL_TTL

    arcadia_build_path = 'yabs/analytics/reach/daily_distribution/daily_distribution'
    task_name = 'reach_daily_distribution'
    task_executable_resource_description = 'Yabs analytics reach_daily_distribution executable binaries'

    class TaskParams(YabsAnalyticsBaseTaskParams):
        config_path = rope.parameters.StrParam('Config path',
                                               default='yabs/analytics/reach/daily_distribution/conf.yaml')
        output_folder = rope.parameters.StrParam('Output folder',
                                                 default='//home/bs/users/yabs-analytics/reach/daily_distribution',
                                                 required=True)
        start_date = rope.parameters.DateParam('Start date',
                                               details='YYYY-MM-DD format (if not set yesterday date will be used)',
                                               default=lambda p: (datetime.now() - timedelta(days=1)).date())
        end_date = rope.parameters.DateParam('End date',
                                             details='YYYY-MM-DD format (if not set yesterday date will be used)',
                                             default=lambda p: (datetime.now() - timedelta(days=1)).date())


@register_as_yabs_analytics_task
class ReachMediaAdMoneyInventory(sdk2.Resource):
    executable = True
    releasers = YABS_ANALYTICS_RELEASERS
    ttl = GENERAL_TTL

    arcadia_build_path = 'yabs/analytics/reach/media_ad_money_inventory/media_ad_money_inventory'
    task_name = 'reach_media_ad_money_inventory'
    task_executable_resource_description = 'Yabs analytics reach_media_ad_money_inventory executable binaries'

    class TaskParams(YabsAnalyticsBaseTaskParams):
        config_path = rope.parameters.StrParam('Config path',
                                               default='yabs/analytics/reach/media_ad_money_inventory/conf.yaml')
        output_folder = rope.parameters.StrParam('Output folder',
                                                 default='//home/bs/users/yabs-analytics/reach/media_ad_money_inventory',
                                                 required=True)
        start_date = rope.parameters.DateParam('Start date',
                                               details='YYYY-MM-DD format (if not set yesterday date will be used)',
                                               default=lambda p: (datetime.now() - timedelta(days=1)).date())
        end_date = rope.parameters.DateParam('End date',
                                             details='YYYY-MM-DD format (if not set yesterday date will be used)',
                                             default=lambda p: (datetime.now() - timedelta(days=1)).date())


@register_as_yabs_analytics_task
class ReachMonitoring(sdk2.Resource):
    executable = True
    releasers = YABS_ANALYTICS_RELEASERS
    ttl = GENERAL_TTL

    arcadia_build_path = 'yabs/analytics/reach/monitoring/monitoring'
    task_name = 'reach_monitoring'
    task_executable_resource_description = 'Yabs analytics reach_monitoring executable binaries'

    class TaskParams(YabsAnalyticsBaseTaskParams):
        config_path = rope.parameters.StrParam('Config path',
                                               default='yabs/analytics/reach/monitoring/conf.yaml')
        output_folder = rope.parameters.StrParam('Output folder',
                                                 default='//home/bs/users/yabs-analytics/reach/monitoring',
                                                 required=True)
        start_date = rope.parameters.DateParam('Start date',
                                               details='YYYY-MM-DD format (if not set yesterday date will be used)',
                                               default=lambda p: (datetime.now() - timedelta(days=1)).date())
        end_date = rope.parameters.DateParam('End date',
                                             details='YYYY-MM-DD format (if not set yesterday date will be used)',
                                             default=lambda p: (datetime.now() - timedelta(days=1)).date())


@register_as_yabs_analytics_task
class ReachReachFrequency(sdk2.Resource):
    executable = True
    releasers = YABS_ANALYTICS_RELEASERS
    ttl = GENERAL_TTL

    arcadia_build_path = 'yabs/analytics/reach/reach_frequency/reach_frequency'
    task_name = 'reach_reach_frequency'
    task_executable_resource_description = 'Yabs analytics reach_reach_frequency executable binaries'

    class TaskParams(YabsAnalyticsBaseTaskParams):
        config_path = rope.parameters.StrParam('Config path',
                                               default='yabs/analytics/reach/reach_frequency/conf.yaml')
        output_folder = rope.parameters.StrParam('Output folder',
                                                 default='//home/bs/users/yabs-analytics/reach/reach_frequency',
                                                 required=True)
        start_date = rope.parameters.DateParam('Start date',
                                               details='YYYY-MM-DD format (if not set yesterday date will be used)',
                                               default=lambda p: (datetime.now() - timedelta(days=1)).date())
        end_date = rope.parameters.DateParam('End date',
                                             details='YYYY-MM-DD format (if not set yesterday date will be used)',
                                             default=lambda p: (datetime.now() - timedelta(days=1)).date())


@register_as_yabs_analytics_task
class ReachVerificatorDashboard(sdk2.Resource):
    executable = True
    releasers = YABS_ANALYTICS_RELEASERS
    ttl = GENERAL_TTL

    arcadia_build_path = 'yabs/analytics/reach/verificator_dashboard/verificator_dashboard'
    task_name = 'reach_verificator_dashboard'
    task_executable_resource_description = 'Yabs analytics reach_verificator_dashboard executable binaries'

    class TaskParams(YabsAnalyticsBaseTaskParams):
        config_path = rope.parameters.StrParam('Config path',
                                               default='yabs/analytics/reach/verificator_dashboard/conf.yaml')
        output_folder = rope.parameters.StrParam('Output folder',
                                                 default='//home/bs/users/yabs-analytics/reach/verificator_dashboard',
                                                 required=True)
        start_date = rope.parameters.DateParam('Start date',
                                               details='YYYY-MM-DD format (if not set yesterday date will be used)',
                                               default=lambda p: (datetime.now() - timedelta(days=1)).date())
        end_date = rope.parameters.DateParam('End date',
                                             details='YYYY-MM-DD format (if not set yesterday date will be used)',
                                             default=lambda p: (datetime.now() - timedelta(days=1)).date())
